<?php

declare(strict_types=1);

namespace Drupal\ip_limiter\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Configure Ip limiter settings for this site.
 */
final class IpLimiterSettingsForm extends ConfigFormBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ip_limiter_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ip_limiter.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('ip_limiter.settings');
    $form['max_requests'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Requests'),
      '#min' => 1,
      '#default_value' => $config->get('max_requests'),
      '#description' => $this->t('The number of allowed requests before blocking the user.'),
    ];

    $form['time_period'] = [
      '#type' => 'number',
      '#title' => $this->t('Time Period (seconds)'),
      '#min' => 1,
      '#default_value' => $config->get('time_period'),
      '#description' => $this->t('The time window in seconds.'),
    ];

    $form['blocked_paths'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Protected Page Paths'),
      '#default_value' => $config->get('blocked_paths'),
      '#description' => $this->t('Enter one path per line (e.g., "my-page"). No leading slash.'),
    ];

    $form['blocked_routes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Protected Route Paths'),
      '#default_value' => $config->get('blocked_routes'),
      '#description' => $this->t('Enter one route per line (e.g., "entity.node.canonical")'),
    ];

    $form['global_restrict'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Restrict Globally'),
      '#default_value' => $config->get('global_restrict'),
      '#description' => $this->t('If checked, users will be blocked from the entire site instead of just the restricted pages.'),
    ];

    $form['response_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Response Type'),
      '#default_value' => $config->get('response_type'),
      '#options' => [
        '403' => $this->t('Access Denied (403)'),
        '404' => $this->t('Page Not Found (404)'),
      ],
      '#description' => $this->t('Select whether blocked users receive a 403 (Access Denied) or 404 (Page Not Found).'),
    ];

    $form['ban_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Ban Duration (seconds)'),
      '#min' => 1,
      '#default_value' => $config->get('ban_duration'),
      '#description' => $this->t('The duration (in seconds) that a user remains banned.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('ip_limiter.settings')
      ->set('max_requests', $form_state->getValue('max_requests'))
      ->set('time_period', $form_state->getValue('time_period'))
      ->set('blocked_paths', $form_state->getValue('blocked_paths'))
      ->set('blocked_routes', $form_state->getValue('blocked_routes'))
      ->set('global_restrict', $form_state->getValue('global_restrict'))
      ->set('response_type', $form_state->getValue('response_type'))
      ->set('ban_duration', $form_state->getValue('ban_duration'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
