<?php

declare(strict_types=1);

namespace Drupal\ip_limiter\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for unbanning an IP address.
 */
final class IpLimiterUnbanConfirmForm extends ConfirmFormBase {

  public function __construct(protected Connection $connection) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static($container->get('database'));
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ip_limiter_unban_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to unban this IP address?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('ip_limiter.bans');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $ip_address = $this->getRouteMatch()->getParameter('ip_address');
    if (empty($ip_address)) {
      return;
    }

    $this->connection->delete('ip_limiter_ban')
      ->condition('ip_address', $ip_address)
      ->execute();

    $this->messenger()->addMessage($this->t('The IP address %ip_address has been unbanned.', ['%ip_address' => $ip_address]));
    $form_state->setRedirect('ip_limiter.bans');
  }

}
