<?php

declare(strict_types=1);

namespace Drupal\Tests\ip_limiter\Kernel;

use Drupal\Component\Datetime\Time;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\ip_limiter\EventSubscriber\IpLimiterSubscriber;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests the IP Limiter event subscriber.
 *
 * @group ip_limiter
 */
class IpLimiterSubscriberTest extends KernelTestBase {

  use NodeCreationTrait;
  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['system', 'node', 'user', 'ip_limiter'];

  /**
   * The IP Limiter subscriber.
   */
  protected IpLimiterSubscriber $subscriber;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The database connection.
   */
  protected Connection $connection;

  /**
   * The time service.
   */
  protected Time $time;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install the user module schema and config.
    $this->installConfig(['user']);
    $this->installEntitySchema('user');
    $this->installEntitySchema('user_role');
    $this->installSchema('ip_limiter', ['ip_limiter_log', 'ip_limiter_ban']);
    $this->installConfig('ip_limiter');
    $this->installConfig(['system']);
    $this->installEntitySchema('node');

    $user = $this->createUser(['access content']);
    // Login the user.
    $this->container->get('current_user')->setAccount($user);

    // Create the node bundles required for testing.
    $type = NodeType::create([
      'type' => 'page',
      'name' => 'page',
    ]);
    $type->save();

    // Create nodes.
    $this->createNode(['type' => 'page', 'title' => 'Page 1']);
    $this->createNode(['type' => 'page', 'title' => 'Page 2']);

    // Initialize the settings.
    $config = $this->config('ip_limiter.settings');
    $config->set('time_period', 5)
      ->set('ban_duration', 5)
      // The max requests is set to 3 so the user will be blocked on the 3rd
      // request.
      ->set('max_requests', 3)
      ->set('blocked_paths', "node/1")
      ->save();

    $this->configFactory = $this->container->get('config.factory');
    $this->connection = $this->container->get('database');
    $this->time = $this->container->get('datetime.time');

    $this->subscriber = new IpLimiterSubscriber(
      $this->configFactory,
      $this->connection,
      $this->time
    );
  }

  /**
   * Performs a request and asserts the expected response code.
   */
  protected function performAndAssertRequest(string $path, int $expectedStatusCode): void {
    // Actually visit the page.
    $request = Request::create($path);
    $response = $this->container->get('http_kernel')->handle($request);

    // Assert the response code.
    $this->assertEquals($expectedStatusCode, $response->getStatusCode());
  }

  /**
   * Tests the IP Limiter functionality.
   *
   * @group ip_limiter
   * @covers \Drupal\ip_limiter\EventSubscriber\IpLimiterSubscriber::onKernelRequest
   */
  public function testIpLimiter(): void {
    // Perform 1st visit to node/1.
    $this->performAndAssertRequest('/node/1', 200);

    // Assert a log entry has been created.
    $logCount = $this->connection->select('ip_limiter_log', 'l')
      ->condition('l.ip_address', Request::create('/node/1')->getClientIp())
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(1, $logCount);

    // Perform 2nd visit to node/1.
    $this->performAndAssertRequest('/node/1', 200);

    // Perform 3rd visit to node/1.
    $this->performAndAssertRequest('/node/1', IpLimiterSubscriber::RESPONSE_TYPE_403);

    // Assert that a ban entry exists and the ban time is 5 seconds.
    $banEntry = $this->connection->select('ip_limiter_ban', 'b')
      ->fields('b', ['ban_end'])
      ->condition('b.ip_address', Request::create('/node/1')->getClientIp())
      ->execute()
      ->fetchAssoc();
    $this->assertNotEmpty($banEntry);
    $this->assertEquals($this->time->getRequestTime() + 5, $banEntry['ban_end']);

    // Visit node/2.
    $this->performAndAssertRequest('/node/2', 200);

    // Set the ban end time to 1 hour in the past.
    $this->connection->update('ip_limiter_ban')
      ->fields(['ban_end' => $this->time->getRequestTime() - 3600])
      ->condition('ip_address', Request::create('/node/1')->getClientIp())
      ->execute();

    // Perform 2 more visits to node/1.
    $this->performAndAssertRequest('/node/1', 200);
    $this->performAndAssertRequest('/node/1', 200);

    // Perform another visit.
    $this->performAndAssertRequest('/node/1', 403);

    // Assert that the ban time is 10 seconds.
    $banEntry = $this->connection->select('ip_limiter_ban', 'b')
      ->fields('b', ['ban_end'])
      ->condition('b.ip_address', Request::create('/node/1')->getClientIp())
      ->execute()
      ->fetchAssoc();
    $this->assertEquals($this->time->getRequestTime() + 10, $banEntry['ban_end']);

    // Update the settings to return a 404 instead.
    $this->config('ip_limiter.settings')
      ->set('response_type', IpLimiterSubscriber::RESPONSE_TYPE_404)
      ->save();

    // Visit the page.
    $this->performAndAssertRequest('/node/1', IpLimiterSubscriber::RESPONSE_TYPE_404);

    // Assert that the ban time is 10 seconds.
    $banEntry = $this->connection->select('ip_limiter_ban', 'b')
      ->fields('b', ['ban_end'])
      ->condition('b.ip_address', Request::create('/node/1')->getClientIp())
      ->execute()
      ->fetchAssoc();
    $this->assertEquals($this->time->getRequestTime() + 10, $banEntry['ban_end']);

    // Update the settings to return a 404 instead.
    $this->config('ip_limiter.settings')
      ->set('response_type', IpLimiterSubscriber::RESPONSE_TYPE_429)
      ->save();

    // Visit the page.
    $this->performAndAssertRequest('/node/1', IpLimiterSubscriber::RESPONSE_TYPE_429);

    // Update the settings to return a 404 instead.
    $this->config('ip_limiter.settings')
      ->set('response_type', IpLimiterSubscriber::RESPONSE_TYPE_403)
      ->save();

    // Visit the page.
    $this->performAndAssertRequest('/node/1', IpLimiterSubscriber::RESPONSE_TYPE_403);

    // Delete the ban entry.
    $this->connection->delete('ip_limiter_ban')
      ->condition('ip_address', Request::create('/node/1')->getClientIp())
      ->execute();

    // Assert that the page can be accessed again.
    $this->performAndAssertRequest('/node/1', 200);
  }

  /**
   * Tests the IP Limiter functionality.
   *
   * @group ip_limiter
   * @covers ::ip_limiter_cron
   */
  public function testCron(): void {
    // Insert a ban entry with a multiplier greater than 1.
    $this->connection->insert('ip_limiter_ban')
      ->fields([
        'ip_address' => '127.0.0.1',
        'multiplier' => 2,
        'created' => $this->time->getRequestTime() - 7200,
        'updated' => $this->time->getRequestTime() - 7200,
        'ban_end' => $this->time->getRequestTime() - 3600,
      ])
      ->execute();

    // Insert a ban entry with a multiplier of 1.
    $this->connection->insert('ip_limiter_ban')
      ->fields([
        'ip_address' => '127.0.0.2',
        'multiplier' => 1,
        'created' => $this->time->getRequestTime() - 7200,
        'updated' => $this->time->getRequestTime() - 7200,
        'ban_end' => $this->time->getRequestTime() - 3600,
      ])
      ->execute();

    // Simulate the cron run.
    ip_limiter_cron();

    // Assert that the multiplier for the first entry has been reduced.
    $banEntry = $this->connection->select('ip_limiter_ban', 'b')
      ->fields('b', ['multiplier'])
      ->condition('b.ip_address', '127.0.0.1')
      ->execute()
      ->fetchAssoc();
    $this->assertEquals(1, $banEntry['multiplier']);

    // Assert that the second entry has been removed.
    $banCount = $this->connection->select('ip_limiter_ban', 'b')
      ->condition('b.ip_address', '127.0.0.2')
      ->countQuery()
      ->execute()
      ->fetchField();
    $this->assertEquals(0, $banCount);
  }

}
