<?php

namespace Drupal\ip_login\Form;

use Drupal\Core\DrupalKernelInterface;
use Drupal\Core\Cache\CacheFactoryInterface;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure ip_login settings.
 */
class IpLoginSettingsForm extends ConfigFormBase {

  /**
   * The cache factory.
   */
  protected CacheFactoryInterface $cacheFactory;

  /**
   * The cache tags invalidator.
   */
  protected CacheTagsInvalidatorInterface $cacheTagsInvalidator;

  /**
   * The kernel service.
   */
  protected DrupalKernelInterface $kernel;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->cacheFactory = $container->get('cache_factory');
    $instance->cacheTagsInvalidator = $container->get('cache_tags.invalidator');
    $instance->kernel = $container->get('kernel');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ip_login_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ip_login.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('ip_login.settings');

    $form['auto_login'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Login automatically'),
      '#description' => $this->t('When an anonymous user accesses any page of the site, the module will attempt to log them in automatically.'),
      '#default_value' => $config->get('auto_login'),
    ];
    $form['form_login'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Login form'),
      '#description' => $this->t('An additional action link is added to the user login form.'),
      '#default_value' => $config->get('form_login'),
    ];

    // Allows ip check on certain pages only.
    $form['active_pages'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Path specific IP logins'),
      '#description' => $this->t('<p>Choose specific paths that IP Login will or will not try to log in a user.</p><p><strong>Please note:</strong></p><ul><li>No alias or system path lookup is performed, so you will need to have both paths listed if you want IP logins to happen on both paths  - e.g. alias "/about/this-site" and system path "/node/123" would both need to be added.</li><li>This will not affect users with <code>can log in as another user</code> permission who have already logged out and have a cookie.</li></ul><br />'),
      '#description_display' => 'before',
      '#collapsible' => FALSE,
    ];

    $options = [
      $this->t('Log in by IP on every page except the listed <strong>paths</strong>.'),
      $this->t('Log in by IP only on the listed <strong>paths</strong>.'),
    ];
    $description = $this->t(
      "Enter one page per line as Drupal paths. The '*' character is a wildcard. Example paths are %blog for the blog page and %blog-wildcard for every personal blog. %front is the front page. Note you must also enter system paths (e.g. '/node/123') for aliases if you want IP logins to happen there too.",
      ['%blog' => '/blog', '%blog-wildcard' => '/blog/*', '%front' => '<front>'],
    );

    $check_mode = $config->get('check_mode') ?? 0;

    $paths = $config->get('paths');
    $form['active_pages']['paths'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Paths'),
      '#default_value' => $paths,
      '#description' => $description,
    ];
    $form['active_pages']['check_mode'] = [
      '#type' => 'radios',
      '#options' => $options,
      '#default_value' => $check_mode,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $config = $this->config('ip_login.settings');
    $form_state->cleanValues();

    $auto_login = $form_state->getValue('auto_login');
    $form_login = $form_state->getValue('form_login');
    $check_mode = $form_state->getValue('check_mode');
    $paths = $form_state->getValue('paths');

    $old_auto_login = $config->get('auto_login');
    $config->set('auto_login', $auto_login);
    $old_form_login = $config->get('form_login');
    $config->set('form_login', $form_login);
    $old_check_mode = $config->get('check_mode');
    $config->set('check_mode', $check_mode);
    $old_paths = $config->get('paths');
    $config->set('paths', $paths);

    // If the login mode is changed, we need to clear the render cache so the
    // auto-login can be attempted on subsequent page loads, if configured.
    if ($auto_login != $old_auto_login || $form_login != $old_form_login || $check_mode != $old_check_mode || $paths != $old_paths) {
      $this->cacheFactory->get('render')->deleteAll();
      $this->cacheTagsInvalidator->invalidateTags(['ip_login']);

      // We need to invalidate the container so the middleware services are
      // registered based on the settings above.
      // @see \Drupal\ip_login\IpLoginServiceProvider::register()
      $this->kernel->invalidateContainer();
    }

    $config->save();
  }

}
