<?php

declare(strict_types=1);

namespace Drupal\ip_login\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\Session\AccountProxy;
use Drupal\Core\Url;
use Drupal\ip_login\IpLoginController;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides the active search facets block.
 *
 * @Block(
 *   id = "ip_login_block",
 *   admin_label = @Translation("IP Login"),
 *   category = @Translation("System"),
 * )
 */
final class IpLoginBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs the plugin instance.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly RequestStack $requestStack,
    private readonly AccountProxy $currentUser,
    private readonly RedirectDestinationInterface $redirectDestination,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('request_stack'),
      $container->get('current_user'),
      $container->get('redirect.destination'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $build = [];
    $request = $this->requestStack->getCurrentRequest();
    $matched_uid = IpLoginController::checkIpLoginExists($request);
    if ($this->currentUser->isAuthenticated() || !$matched_uid) {
      return $build;
    }
    else {
      $build['content']['ip_login'] = [
        '#type' => 'link',
        '#title' => $this->t('Log in automatically using IP'),
        '#url' => Url::fromRoute('ip_login.dologin', [], [
          'query' => $this->redirectDestination->getAsArray(),
        ]),
      ];
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return 0;
  }

}
