<?php

namespace Drupal\ironstar\Compiler;

use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;

/**
 * Configures monolog for the Ironstar platform.
 */
class MonologCompilerPass implements CompilerPassInterface {

  /**
   * {@inheritdoc}
   */
  public function process(ContainerBuilder $container): void {
    $config = $container->get('config.storage');

    $monolog_enabled = isset($container->getParameter('container.modules')['monolog']);
    $handler_name = 'ironstar_drupal';
    $drupal_log_definition = 'monolog.handler.' . $handler_name;
    $monolog_config = [
      'enable' => $_ENV['IRONSTAR_DRUPAL_ENABLE_MONOLOG'] ?? $config->read('ironstar.settings')['monolog']['enabled'] ?? NULL,
      'log_level' => $_ENV['IRONSTAR_DRUPAL_LOG_LEVEL'] ?? $config->read('ironstar.settings')['monolog']['log_level'] ?? 'NOTICE',
      /* Unlikely to ever change but adding as an option to allow changing the path without requiring a new deployment. */
      /* This should only ever be set if recommended by Ironstar support. */
      'path' => $_ENV['IRONSTAR_DRUPAL_LOG_PATH'] ?? '/app/logs/drupal.log',
    ];
    $supported_log_levels = [
      'DEBUG',
      'INFO',
      'NOTICE',
      'WARNING',
      'ERROR',
      'CRITICAL',
      'ALERT',
      'EMERGENCY',
    ];

    /* Don't configure drupal.log if monolog is disabled or the config option is disabled by the user. */
    if (!$monolog_enabled || (isset($monolog_config['enable']) && !$monolog_config['enable'])) {
      return;
    }

    /* Remove any conflicting service definitions that would result in duplicate logs. */
    foreach ($container->getDefinitions() as $id => $definition) {
      if (in_array($monolog_config['path'], $definition->getArguments()) && str_contains($id, 'monolog.handler')) {
        $container->removeDefinition($id);
      }
    }

    /* Override Log Level if an invalid value is set in the environment variable. */
    if (!in_array(strtoupper($monolog_config['log_level']), $supported_log_levels)) {
      $monolog_config['log_level'] = 'NOTICE';
    }

    $definition = new Definition(
      'Monolog\Handler\StreamHandler',
      [$monolog_config['path'], $monolog_config['log_level']],
    );
    $definition->setPublic(TRUE);

    $existing_channel_handlers = $container->getParameter('monolog.channel_handlers');
    $existing_processors = $container->getParameter('monolog.processors');

    /* Don't override any existing monolog configuration. */
    // phpcs:ignore DrupalPractice.CodeAnalysis.VariableAnalysis.UnusedVariable
    foreach ($existing_channel_handlers as $unused => &$handlers) {
      array_push($handlers, $handler_name);
    }
    $existing_processors = array_merge($existing_processors, [
      'message_placeholder',
      'current_user',
      'request_uri',
      'ip',
      'referer',
      'filter_backtrace',
      'memory_usage',
    ]);
    $existing_processors = array_unique($existing_processors);

    $container->setParameter('monolog.channel_handlers', $existing_channel_handlers);
    $container->setParameter('monolog.processors', $existing_processors);
    $container->setDefinition($drupal_log_definition, $definition);
  }

}
