<?php

namespace Drupal\ironstar\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class PurgeQueueStatus Defines a controller to return purge queue stats.
 */
class PurgeQueueStatus extends ControllerBase {
  /**
   * The purge queue stats tracker service.
   *
   * @var Drupal\purge\Plugin\Purge\Queue\StatsTracker
   */
  protected $stats;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): PurgeQueueStatus {
    /** @var \Drupal\Core\Extension\ModuleHandler $module_handler */
    $module_handler = $container->get('module_handler');

    $instance = parent::create($container);

    if ($module_handler->moduleExists('purge')) {
      $instance->stats = $container->get('purge.queue.stats');
    }

    return $instance;
  }

  /**
   * Returns purge queue stats using the purge.queue.stats service.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON Response containing all purge queue stats similar to the
   *   drush p:queue-stats command.
   */
  public function get(): JsonResponse {
    if ($this->stats) {
      $resp = new JsonResponse([
        'time' => date('Y-m-d H:i:s'),
        'number_of_items' => $this->stats->numberOfItems()->getInteger(),
        'total_succeeded' => $this->stats->totalSucceeded()->getInteger(),
        'total_failed' => $this->stats->totalFailed()->getInteger(),
        'total_processing' => $this->stats->totalProcessing()->getInteger(),
        'total_not_supported' => $this->stats->totalNotSupported()->getInteger(),
      ], 200);
    }
    else {
      $resp = new JsonResponse("missing purge module", 200);
    }

    return $resp;
  }

}
