<?php

namespace Drupal\ironstar\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class IronstarSettingsForm Defines a form to configure module settings.
 */
class IronstarSettingsForm extends ConfigFormBase {
  /**
   * Whether the monolog module is enabled.
   *
   * @var bool
   */
  protected $isEnabled;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): IronstarSettingsForm {
    /** @var \Drupal\Core\Extension\ModuleHandler $module_handler */
    $module_handler = $container->get('module_handler');

    $instance = parent::create($container);
    $instance->isEnabled = $module_handler->moduleExists('monolog');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ironstar_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ironstar.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    if (!$this->isEnabled) {
      $this->messenger()->addWarning($this->t("The monolog module is not currently installed or enabled so it can't be configured"));
    }

    $config = $this->config('ironstar.settings');
    $enable_monolog = getenv('IRONSTAR_DRUPAL_ENABLE_MONOLOG') !== FALSE ? getenv('IRONSTAR_DRUPAL_ENABLE_MONOLOG') : NULL ?? $config->get('monolog')['enabled'] ?? $this->isEnabled;
    $log_level = getenv('IRONSTAR_DRUPAL_LOG_LEVEL') !== FALSE ? getenv('IRONSTAR_DRUPAL_LOG_LEVEL') : NULL ?? $config->get('monolog')['log_level'] ?? 'NOTICE';
    $supported_log_levels = [
      'DEBUG',
      'INFO',
      'NOTICE',
      'WARNING',
      'ERROR',
      'CRITICAL',
      'ALERT',
      'EMERGENCY',
    ];
    $enable_monolog_overridden = FALSE;
    $log_level_overridden = FALSE;

    $form['monolog'] = [
      '#type' => 'details',
      '#title' => $this->t('Monolog configuration'),
      '#open' => TRUE,
    ];

    if (getenv('IRONSTAR_DRUPAL_ENABLE_MONOLOG') !== FALSE) {
      $enable_monolog_overridden = TRUE;
      $this->messenger()->addWarning($this->t('Enabling monolog configuration is overridden by environment variable: IRONSTAR_DRUPAL_ENABLE_MONOLOG'));
    }

    if (getenv('IRONSTAR_ENV_NAME') !== FALSE) {
      $log_path = '/app/logs/drupal.log';
    }
    elseif (getenv('DDEV_VERSION') !== FALSE) {
      $log_path = '/var/log/drupal.log';
    }
    else {
      $log_path = NULL;
    }
    if (!file_exists(dirname(strval($log_path)))) {
      $enable_monolog_overridden = FALSE;
      $this->isEnabled = FALSE;
      $this->messenger()->addWarning($this->t("The monolog configuration is disabled because a valid log path can't be found"));
    }

    $form['monolog']['enable_monolog'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Configure <b>drupal.log</b> using Ironstar defaults'),
      '#default_value' => $enable_monolog,
      '#disabled' => !$this->isEnabled || $enable_monolog_overridden,
    ];

    if (!in_array(strtoupper($log_level), $supported_log_levels)) {
      $log_level_overridden = TRUE;
      $this->messenger()->addWarning($this->t('Invalid Log Level set in environment variable: IRONSTAR_DRUPAL_LOG_LEVEL<br>Using default Log Level: Notice'));
      $log_level = 'NOTICE';
    }
    if (getenv('IRONSTAR_DRUPAL_LOG_LEVEL') !== FALSE) {
      $log_level_overridden = TRUE;
      $this->messenger()->addWarning($this->t('Log Level is overridden by environment variable: IRONSTAR_DRUPAL_LOG_LEVEL'));
    }

    $log_level = strtoupper($log_level);
    $form['monolog']['log_level'] = [
      '#type' => 'select',
      '#title' => $this->t('Log Level'),
      '#default_value' => $log_level,
      '#options' => [
        'DEBUG' => $this->t('Debug'),
        'INFO' => $this->t('Info'),
        'NOTICE' => $this->t('Notice'),
        'WARNING' => $this->t('Warning'),
        'ERROR' => $this->t('Error'),
        'CRITICAL' => $this->t('Critical'),
        'ALERT' => $this->t('Alert'),
        'EMERGENCY' => $this->t('Emergency'),
      ],
      '#disabled' => !$this->isEnabled || $log_level_overridden,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('ironstar.settings')
      ->set('monolog.enabled', $form_state->getValue('enable_monolog'))
      ->set('monolog.log_level', $form_state->getValue('log_level'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
