<?php

namespace Drupal\issue_queue\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;

/**
 * Provides a 'Issue Queue' form.
 */
class IssueForm extends FormBase {
  use LoggerChannelTrait;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * IssueForm constructor.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    MessengerInterface $messenger,
  ) {
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('messenger'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'issue_queue_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $path = NULL) {

    $form['title'] = [
      '#type' => 'markup',
      '#markup' => '<p class="issue-queue-title" title=Please report any issues you find so we can fix them. Thanks!">Please report any issues you find so we can fix them. Thanks!</p>',
    ];

    $form['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Page'),
      '#default_value' => $form_state->getValue('path') ?? $path,
      '#attributes' => [
        'class' => ['issue-queue-textfield'],
        'readonly' => 'readonly',
      ],
    ];

    $form['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#description' => $this->t('Enter type of issue.'),
      '#default_value' => $form_state->getValue('subject') ?? '',
      '#attributes' => [
        'class' => ['issue-queue-textfield'],
      ],
    ];

    $form['issue'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Issue'),
      '#description' => $this->t('Enter issue information.'),
      '#attributes' => [
        'class' => ['issue-queue-textarea'],
      ],
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if (empty($form_state->getValue('subject'))) {
      $form_state->setErrorByName('subject', $this->t('Subject cannot be empty.'));
    }
    if (empty($form_state->getValue('issue'))) {
      $form_state->setErrorByName('issue', $this->t('Issue cannot be empty.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $path = $form_state->getValue('path');
    $subject = $form_state->getValue('subject');
    $issue = $form_state->getValue('issue');
    $this->getLogger('issue_queue')->info('@subject - @issue <a href=":path?issue=' . $issue . '">Link to page.</a>.', [
      '@subject' => $subject,
      '@issue' => $issue,
      ':path' => $path,
    ]);
    $this->messenger->addStatus($this->t('Thank you for submitting this issue.'));
  }

}
