<?php

namespace Drupal\jellyfin_integration\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\jellyfin_integration\Service\JellyfinClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying Jellyfin library content.
 */
class JellyfinLibraryController extends ControllerBase
{

    /**
     * The Jellyfin client.
     *
     * @var \Drupal\jellyfin_integration\Service\JellyfinClient
     */
    protected $jellyfinClient;

    /**
     * Constructs a JellyfinLibraryController object.
     *
     * @param \Drupal\jellyfin_integration\Service\JellyfinClient $jellyfin_client
     *   The Jellyfin client service.
     */
    public function __construct(JellyfinClient $jellyfin_client)
    {
        $this->jellyfinClient = $jellyfin_client;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('jellyfin_integration.client')
        );
    }

    /**
     * Displays the library overview.
     *
     * @return array
     *   Render array.
     */
    public function overview()
    {
        $libraries = $this->jellyfinClient->getLibraries();

        if ($libraries === NULL) {
            $this->messenger()->addError($this->t('Failed to fetch libraries from Jellyfin. Please check your configuration.'));
            return [
                '#markup' => $this->t('Unable to connect to Jellyfin server.'),
            ];
        }

        if (empty($libraries)) {
            return [
                '#markup' => $this->t('No libraries found in Jellyfin.'),
            ];
        }

        $header = [
            $this->t('Name'),
            $this->t('Type'),
            $this->t('Locations'),
        ];

        $rows = [];
        foreach ($libraries as $library) {
            $name = $library['Name'] ?? 'Unknown';
            $type = $library['CollectionType'] ?? 'mixed';
            $locations = $library['Locations'] ?? [];

            $url = Url::fromRoute('jellyfin_integration.library_items', ['library_id' => $library['ItemId'] ?? '']);
            $link = Link::fromTextAndUrl($name, $url)->toString();

            $rows[] = [
                $link,
                ucfirst($type),
                implode(', ', $locations),
            ];
        }

        return [
            'status_messages' => [
                '#type' => 'status_messages',
                '#weight' => -100,
            ],
            'help_text' => [
                '#type' => 'container',
                '#attributes' => ['class' => ['description', 'messages', 'messages--status']],
                '#markup' => $this->t('<strong>Note:</strong> Click on a library name to browse its content. Libraries are configured in the Jellyfin server.'),
                '#weight' => -90,
            ],
            'table' => [
                '#type' => 'table',
                '#header' => $header,
                '#rows' => $rows,
                '#empty' => $this->t('No libraries found.'),
            ],
        ];
    }

    /**
     * Displays items in a specific library.
     *
     * @param string $library_id
     *   The library ID.
     *
     * @return array
     *   Render array.
     */
    public function libraryItems($library_id)
    {
        $items = $this->jellyfinClient->getItems([
            'ParentId' => $library_id,
            'Limit' => 50,
            'Fields' => 'Overview,ProductionYear',
        ]);

        if ($items === NULL) {
            $this->messenger()->addError($this->t('Failed to fetch items from Jellyfin.'));
            return [
                '#markup' => $this->t('Unable to load library content.'),
            ];
        }

        $media_items = $items['Items'] ?? [];

        if (empty($media_items)) {
            return [
                '#markup' => $this->t('No items found in this library.'),
            ];
        }

        $build = [
            'status_messages' => [
                '#type' => 'status_messages',
                '#weight' => -100,
            ],
            'grid' => [
                '#type' => 'container',
                '#attributes' => ['class' => ['jellyfin-library-grid']],
            ],
        ];

        foreach ($media_items as $item) {
            $item_id = $item['Id'] ?? '';
            $name = $item['Name'] ?? 'Unknown';
            $year = $item['ProductionYear'] ?? '';
            $type = $item['Type'] ?? 'Unknown';

            $poster_url = $this->jellyfinClient->getPosterUrl($item_id, 200);

            $url = Url::fromRoute('jellyfin_integration.item_detail', ['item_id' => $item_id]);

            $build['grid']['items'][$item_id] = [
                '#type' => 'container',
                '#attributes' => ['class' => ['jellyfin-item']],
                'link' => [
                    '#type' => 'link',
                    '#title' => [
                        'image' => [
                            '#type' => 'html_tag',
                            '#tag' => 'img',
                            '#attributes' => [
                                'src' => $poster_url,
                                'alt' => $name,
                                'style' => 'max-width: 200px; height: auto;',
                                'loading' => 'lazy',
                            ],
                        ],
                        'title' => [
                            '#type' => 'html_tag',
                            '#tag' => 'div',
                            '#value' => $name . ($year ? ' (' . $year . ')' : ''),
                            '#attributes' => ['class' => ['jellyfin-item-title']],
                        ],
                    ],
                    '#url' => $url,
                ],
            ];
        }

        $build['#attached']['library'][] = 'jellyfin_integration/library-grid';

        return $build;
    }

    /**
     * Gets the title for a library.
     *
     * @param string $library_id
     *   The library ID.
     *
     * @return string
     *   The library title.
     */
    public function libraryTitle($library_id)
    {
        $item = $this->jellyfinClient->getItem($library_id);
        return $item['Name'] ?? $this->t('Library');
    }

    /**
     * Displays details of a specific item.
     *
     * @param string $item_id
     *   The item ID.
     *
     * @return array
     *   Render array.
     */
    public function itemDetail($item_id)
    {
        $item = $this->jellyfinClient->getItem($item_id);

        if ($item === NULL) {
            $this->messenger()->addError($this->t('Failed to fetch item details from Jellyfin.'));
            return [
                '#markup' => $this->t('Unable to load item details.'),
            ];
        }

        $name = $item['Name'] ?? 'Unknown';
        $year = $item['ProductionYear'] ?? '';
        $overview = $item['Overview'] ?? '';
        $runtime = $item['RunTimeTicks'] ?? 0;
        $genres = $item['Genres'] ?? [];
        $rating = $item['CommunityRating'] ?? '';

        // Convert runtime from ticks to minutes.
        $runtime_minutes = $runtime > 0 ? round($runtime / 600000000) : 0;

        $backdrop_url = $this->jellyfinClient->getBackdropUrl($item_id, 1280);
        $poster_url = $this->jellyfinClient->getPosterUrl($item_id, 300);

        $build = [
            'status_messages' => [
                '#type' => 'status_messages',
                '#weight' => -100,
            ],
            'container' => [
                '#type' => 'container',
                '#attributes' => ['class' => ['jellyfin-item-detail']],
            ],
        ];

        $build['container']['backdrop'] = [
            '#type' => 'html_tag',
            '#tag' => 'div',
            '#attributes' => [
                'class' => ['jellyfin-backdrop'],
                'style' => 'background-image: url(' . $backdrop_url . '); background-size: cover; height: 300px; margin-bottom: 20px;',
            ],
        ];

        $build['container']['content'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['jellyfin-item-content']],
        ];

        $build['container']['content']['poster'] = [
            '#type' => 'html_tag',
            '#tag' => 'img',
            '#attributes' => [
                'src' => $poster_url,
                'alt' => $name,
                'style' => 'float: left; margin-right: 20px; max-width: 300px;',
            ],
        ];

        $build['container']['content']['info'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['jellyfin-item-info']],
        ];

        $build['container']['content']['info']['title'] = [
            '#type' => 'html_tag',
            '#tag' => 'h2',
            '#value' => $name . ($year ? ' (' . $year . ')' : ''),
        ];

        if ($rating) {
            $build['container']['content']['info']['rating'] = [
                '#type' => 'html_tag',
                '#tag' => 'div',
                '#value' => $this->t('Rating: @rating', ['@rating' => round($rating, 1)]),
                '#attributes' => ['class' => ['jellyfin-rating']],
            ];
        }

        if ($runtime_minutes) {
            $build['container']['content']['info']['runtime'] = [
                '#type' => 'html_tag',
                '#tag' => 'div',
                '#value' => $this->t('Runtime: @minutes minutes', ['@minutes' => $runtime_minutes]),
                '#attributes' => ['class' => ['jellyfin-runtime']],
            ];
        }

        if (!empty($genres)) {
            $build['container']['content']['info']['genres'] = [
                '#type' => 'html_tag',
                '#tag' => 'div',
                '#value' => $this->t('Genres: @genres', ['@genres' => implode(', ', $genres)]),
                '#attributes' => ['class' => ['jellyfin-genres']],
            ];
        }

        if ($overview) {
            $build['container']['content']['info']['overview'] = [
                '#type' => 'html_tag',
                '#tag' => 'p',
                '#value' => $overview,
                '#attributes' => ['class' => ['jellyfin-overview']],
            ];
        }

        return $build;
    }

    /**
     * Gets the title for an item detail page.
     *
     * @param string $item_id
     *   The item ID.
     *
     * @return string
     *   The item title.
     */
    public function itemTitle($item_id)
    {
        $item = $this->jellyfinClient->getItem($item_id);
        return $item['Name'] ?? $this->t('Item');
    }

    /**
     * Displays all movies.
     *
     * @return array
     *   Render array.
     */
    public function movies()
    {
        $result = $this->jellyfinClient->getMovies(['Limit' => 50]);

        if ($result === NULL) {
            $this->messenger()->addError($this->t('Failed to fetch movies from Jellyfin.'));
            return [
                '#markup' => $this->t('Unable to load movies.'),
            ];
        }

        $movies = $result['Items'] ?? [];
        $total = $result['TotalRecordCount'] ?? count($movies);

        $build = [
            'status_messages' => [
                '#type' => 'status_messages',
                '#weight' => -100,
            ],
            'summary' => [
                '#type' => 'html_tag',
                '#tag' => 'p',
                '#value' => $this->t('Showing @count of @total movies', [
                    '@count' => count($movies),
                    '@total' => $total,
                ]),
                '#weight' => -90,
            ],
            'grid' => [
                '#type' => 'container',
                '#attributes' => [
                    'class' => ['jellyfin-movie-grid'],
                    'style' => 'display: grid; grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); gap: 20px;',
                ],
            ],
        ];

        foreach ($movies as $movie) {
            $item_id = $movie['Id'] ?? '';
            $name = $movie['Name'] ?? 'Unknown';
            $year = $movie['ProductionYear'] ?? '';

            $poster_url = $this->jellyfinClient->getPosterUrl($item_id, 150);
            $detail_url = Url::fromRoute('jellyfin_integration.item_detail', ['item_id' => $item_id]);

            $build['grid'][$item_id] = [
                '#type' => 'container',
                '#attributes' => ['class' => ['jellyfin-movie-card'], 'style' => 'text-align: center;'],
                'link' => [
                    '#type' => 'link',
                    '#title' => [
                        'image' => [
                            '#type' => 'html_tag',
                            '#tag' => 'img',
                            '#attributes' => [
                                'src' => $poster_url,
                                'alt' => $name,
                                'style' => 'width: 100%; height: auto; border-radius: 8px;',
                                'loading' => 'lazy',
                            ],
                        ],
                    ],
                    '#url' => $detail_url,
                ],
                'title' => [
                    '#type' => 'html_tag',
                    '#tag' => 'div',
                    '#value' => $name . ($year ? ' (' . $year . ')' : ''),
                    '#attributes' => ['style' => 'margin-top: 8px; font-size: 12px;'],
                ],
            ];
        }

        return $build;
    }

}
