<?php

namespace Drupal\jellyfin_integration\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\jellyfin_integration\Service\JellyfinClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Jellyfin Integration settings.
 */
class JellyfinSettingsForm extends ConfigFormBase
{

    /**
     * The Jellyfin client.
     *
     * @var \Drupal\jellyfin_integration\Service\JellyfinClient
     */
    protected $jellyfinClient;

    /**
     * Constructs a JellyfinSettingsForm object.
     *
     * @param \Drupal\jellyfin_integration\Service\JellyfinClient $jellyfin_client
     *   The Jellyfin client service.
     */
    public function __construct(JellyfinClient $jellyfin_client)
    {
        $this->jellyfinClient = $jellyfin_client;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('jellyfin_integration.client')
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function getEditableConfigNames()
    {
        return ['jellyfin_integration.settings'];
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId()
    {
        return 'jellyfin_integration_settings_form';
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state)
    {
        $config = $this->config('jellyfin_integration.settings');

        $form['status_messages'] = [
            '#type' => 'status_messages',
            '#weight' => -100,
        ];

        $form['connection_result'] = [
            '#type' => 'container',
            '#attributes' => ['id' => 'connection-test-result'],
            '#weight' => -90,
        ];

        $form['server_url'] = [
            '#type' => 'url',
            '#title' => $this->t('Jellyfin Server URL'),
            '#description' => $this->t('The URL of your Jellyfin server (e.g., http://localhost:8096). For Docker, use http://host.docker.internal:8096'),
            '#default_value' => $config->get('server_url'),
            '#required' => TRUE,
        ];

        $form['api_key'] = [
            '#type' => 'textfield',
            '#title' => $this->t('API Key'),
            '#description' => $this->t('Generate an API key from Jellyfin Dashboard → API Keys.'),
            '#default_value' => $config->get('api_key'),
            '#required' => TRUE,
        ];

        $form['test_connection'] = [
            '#type' => 'button',
            '#value' => $this->t('Test Connection'),
            '#ajax' => [
                'callback' => '::testConnectionAjax',
                'wrapper' => 'connection-test-result',
                'effect' => 'fade',
            ],
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * AJAX callback for testing the connection.
     */
    public function testConnectionAjax(array &$form, FormStateInterface $form_state)
    {
        $server_url = $form_state->getValue('server_url');
        $api_key = $form_state->getValue('api_key');

        // Temporarily set the config for testing.
        $config = $this->configFactory()->getEditable('jellyfin_integration.settings');
        $original_url = $config->get('server_url');
        $original_key = $config->get('api_key');

        $config->set('server_url', $server_url)
            ->set('api_key', $api_key)
            ->save();

        $result = $this->jellyfinClient->testConnection();

        // Restore original config if different.
        if ($original_url !== $server_url || $original_key !== $api_key) {
            $config->set('server_url', $original_url)
                ->set('api_key', $original_key)
                ->save();
        }

        $element = [
            '#type' => 'container',
            '#attributes' => ['id' => 'connection-test-result'],
        ];

        if ($result) {
            $element['message'] = [
                '#type' => 'html_tag',
                '#tag' => 'div',
                '#attributes' => ['class' => ['messages', 'messages--status']],
                '#value' => $this->t('Successfully connected to Jellyfin server. Server: @name, Version: @version', [
                    '@name' => $result['ServerName'] ?? 'Unknown',
                    '@version' => $result['Version'] ?? 'Unknown',
                ]),
            ];
        } else {
            $element['message'] = [
                '#type' => 'html_tag',
                '#tag' => 'div',
                '#attributes' => ['class' => ['messages', 'messages--error']],
                '#value' => $this->t('Failed to connect to Jellyfin server. Please check your URL and API key.'),
            ];
        }

        return $element;
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        $this->config('jellyfin_integration.settings')
            ->set('server_url', $form_state->getValue('server_url'))
            ->set('api_key', $form_state->getValue('api_key'))
            ->save();

        parent::submitForm($form, $form_state);
    }

}
