<?php

namespace Drupal\jellyfin_integration\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service for interacting with the Jellyfin API.
 */
class JellyfinClient
{

    /**
     * The HTTP client.
     *
     * @var \GuzzleHttp\ClientInterface
     */
    protected $httpClient;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Constructs a JellyfinClient object.
     *
     * @param \GuzzleHttp\ClientInterface $http_client
     *   The HTTP client.
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
     *   The logger factory.
     */
    public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory)
    {
        $this->httpClient = $http_client;
        $this->configFactory = $config_factory;
        $this->logger = $logger_factory->get('jellyfin_integration');
    }

    /**
     * Gets the configured server URL.
     *
     * @return string
     *   The server URL.
     */
    protected function getServerUrl()
    {
        return rtrim($this->configFactory->get('jellyfin_integration.settings')->get('server_url'), '/');
    }

    /**
     * Gets the configured API key.
     *
     * @return string
     *   The API key.
     */
    protected function getApiKey()
    {
        return $this->configFactory->get('jellyfin_integration.settings')->get('api_key');
    }

    /**
     * Makes an API request to Jellyfin.
     *
     * @param string $method
     *   The HTTP method.
     * @param string $endpoint
     *   The API endpoint.
     * @param array $options
     *   Additional request options.
     *
     * @return array|null
     *   The decoded JSON response or NULL on failure.
     */
    protected function request($method, $endpoint, array $options = [])
    {
        $url = $this->getServerUrl() . $endpoint;
        $api_key = $this->getApiKey();

        $options['headers'] = $options['headers'] ?? [];
        $options['headers']['Authorization'] = 'MediaBrowser Token="' . $api_key . '"';
        $options['headers']['Content-Type'] = 'application/json';

        try {
            $response = $this->httpClient->request($method, $url, $options);
            $body = $response->getBody()->getContents();
            return json_decode($body, TRUE);
        } catch (GuzzleException $e) {
            $this->logger->error('Jellyfin API error: @message', ['@message' => $e->getMessage()]);
            return NULL;
        }
    }

    // =========================================================================
    // SYSTEM OPERATIONS
    // =========================================================================

    /**
     * Tests the connection to the Jellyfin server.
     *
     * @return array|null
     *   Server info or NULL on failure.
     */
    public function testConnection()
    {
        return $this->getServerInfo();
    }

    /**
     * Gets server information.
     *
     * @return array|null
     *   Server info or NULL on failure.
     */
    public function getServerInfo()
    {
        return $this->request('GET', '/System/Info');
    }

    /**
     * Gets public server information (no auth required).
     *
     * @return array|null
     *   Public server info or NULL on failure.
     */
    public function getPublicInfo()
    {
        $url = $this->getServerUrl() . '/System/Info/Public';
        try {
            $response = $this->httpClient->request('GET', $url);
            $body = $response->getBody()->getContents();
            return json_decode($body, TRUE);
        } catch (GuzzleException $e) {
            $this->logger->error('Jellyfin API error: @message', ['@message' => $e->getMessage()]);
            return NULL;
        }
    }

    /**
     * Gets server endpoints.
     *
     * @return array|null
     *   Server endpoints or NULL on failure.
     */
    public function getEndpoints()
    {
        return $this->request('GET', '/System/Endpoint');
    }

    // =========================================================================
    // USER OPERATIONS
    // =========================================================================

    /**
     * Gets all users.
     *
     * @return array|null
     *   List of users or NULL on failure.
     */
    public function getUsers()
    {
        return $this->request('GET', '/Users');
    }

    /**
     * Gets a specific user.
     *
     * @param string $user_id
     *   The user ID.
     *
     * @return array|null
     *   User info or NULL on failure.
     */
    public function getUser($user_id)
    {
        return $this->request('GET', '/Users/' . $user_id);
    }

    /**
     * Gets public users.
     *
     * @return array|null
     *   List of public users or NULL on failure.
     */
    public function getPublicUsers()
    {
        return $this->request('GET', '/Users/Public');
    }

    // =========================================================================
    // LIBRARY OPERATIONS
    // =========================================================================

    /**
     * Gets all virtual folders (libraries).
     *
     * @return array|null
     *   List of libraries or NULL on failure.
     */
    public function getLibraries()
    {
        return $this->request('GET', '/Library/VirtualFolders');
    }

    /**
     * Refreshes a library.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function refreshLibrary()
    {
        return $this->request('POST', '/Library/Refresh');
    }

    /**
     * Gets media folders.
     *
     * @return array|null
     *   List of media folders or NULL on failure.
     */
    public function getMediaFolders()
    {
        return $this->request('GET', '/Library/MediaFolders');
    }

    // =========================================================================
    // ITEM OPERATIONS
    // =========================================================================

    /**
     * Queries items with filters.
     *
     * @param array $params
     *   Query parameters.
     *
     * @return array|null
     *   Items result or NULL on failure.
     */
    public function getItems(array $params = [])
    {
        $defaults = [
            'Recursive' => 'true',
            'Limit' => 100,
        ];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/Items', ['query' => $params]);
    }

    /**
     * Gets a specific item.
     *
     * @param string $item_id
     *   The item ID.
     *
     * @return array|null
     *   Item info or NULL on failure.
     */
    public function getItem($item_id)
    {
        // Use bulk endpoint with Ids filter - single item endpoint requires user context.
        $result = $this->request('GET', '/Items', [
            'query' => [
                'Ids' => $item_id,
                'Fields' => 'Overview,Genres,Studios,People,ProductionYear,CommunityRating,OfficialRating,RunTimeTicks',
            ],
        ]);

        // Return the first item if found.
        if ($result && !empty($result['Items'])) {
            return $result['Items'][0];
        }

        return NULL;
    }

    /**
     * Gets items for a specific user.
     *
     * @param string $user_id
     *   The user ID.
     * @param array $params
     *   Query parameters.
     *
     * @return array|null
     *   Items result or NULL on failure.
     */
    public function getUserItems($user_id, array $params = [])
    {
        $defaults = [
            'Recursive' => 'true',
            'Limit' => 100,
        ];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/Users/' . $user_id . '/Items', ['query' => $params]);
    }

    /**
     * Gets movies.
     *
     * @param array $params
     *   Additional query parameters.
     *
     * @return array|null
     *   Movies result or NULL on failure.
     */
    public function getMovies(array $params = [])
    {
        $params['IncludeItemTypes'] = 'Movie';
        return $this->getItems($params);
    }

    /**
     * Gets TV series.
     *
     * @param array $params
     *   Additional query parameters.
     *
     * @return array|null
     *   Series result or NULL on failure.
     */
    public function getSeries(array $params = [])
    {
        $params['IncludeItemTypes'] = 'Series';
        return $this->getItems($params);
    }

    /**
     * Gets episodes.
     *
     * @param array $params
     *   Additional query parameters.
     *
     * @return array|null
     *   Episodes result or NULL on failure.
     */
    public function getEpisodes(array $params = [])
    {
        $params['IncludeItemTypes'] = 'Episode';
        return $this->getItems($params);
    }

    /**
     * Gets latest media items.
     *
     * @param string $user_id
     *   The user ID.
     * @param array $params
     *   Additional query parameters.
     *
     * @return array|null
     *   Latest items or NULL on failure.
     */
    public function getLatestMedia($user_id, array $params = [])
    {
        return $this->request('GET', '/Users/' . $user_id . '/Items/Latest', ['query' => $params]);
    }

    /**
     * Gets resume items (continue watching).
     *
     * @param string $user_id
     *   The user ID.
     * @param array $params
     *   Additional query parameters.
     *
     * @return array|null
     *   Resume items or NULL on failure.
     */
    public function getResumeItems($user_id, array $params = [])
    {
        return $this->request('GET', '/Users/' . $user_id . '/Items/Resume', ['query' => $params]);
    }

    /**
     * Gets similar items.
     *
     * @param string $item_id
     *   The item ID.
     * @param array $params
     *   Additional query parameters.
     *
     * @return array|null
     *   Similar items or NULL on failure.
     */
    public function getSimilarItems($item_id, array $params = [])
    {
        return $this->request('GET', '/Items/' . $item_id . '/Similar', ['query' => $params]);
    }

    // =========================================================================
    // SEARCH OPERATIONS
    // =========================================================================

    /**
     * Searches the library.
     *
     * @param string $query
     *   The search query.
     * @param array $params
     *   Additional query parameters.
     *
     * @return array|null
     *   Search results or NULL on failure.
     */
    public function search($query, array $params = [])
    {
        $params['SearchTerm'] = $query;
        return $this->request('GET', '/Search/Hints', ['query' => $params]);
    }

    // =========================================================================
    // GENRE/CATEGORY OPERATIONS
    // =========================================================================

    /**
     * Gets all genres.
     *
     * @param array $params
     *   Query parameters.
     *
     * @return array|null
     *   List of genres or NULL on failure.
     */
    public function getGenres(array $params = [])
    {
        return $this->request('GET', '/Genres', ['query' => $params]);
    }

    /**
     * Gets all studios.
     *
     * @param array $params
     *   Query parameters.
     *
     * @return array|null
     *   List of studios or NULL on failure.
     */
    public function getStudios(array $params = [])
    {
        return $this->request('GET', '/Studios', ['query' => $params]);
    }

    /**
     * Gets all artists.
     *
     * @param array $params
     *   Query parameters.
     *
     * @return array|null
     *   List of artists or NULL on failure.
     */
    public function getArtists(array $params = [])
    {
        return $this->request('GET', '/Artists', ['query' => $params]);
    }

    /**
     * Gets all persons (actors, directors, etc.).
     *
     * @param array $params
     *   Query parameters.
     *
     * @return array|null
     *   List of persons or NULL on failure.
     */
    public function getPersons(array $params = [])
    {
        return $this->request('GET', '/Persons', ['query' => $params]);
    }

    // =========================================================================
    // IMAGE OPERATIONS
    // =========================================================================

    /**
     * Gets the URL for an item's image.
     *
     * @param string $item_id
     *   The item ID.
     * @param string $type
     *   The image type (Primary, Backdrop, Thumb, etc.).
     * @param array $params
     *   Optional parameters (maxWidth, maxHeight, quality).
     *
     * @return string
     *   The image URL.
     */
    public function getImageUrl($item_id, $type = 'Primary', array $params = [])
    {
        $url = $this->getServerUrl() . '/Items/' . $item_id . '/Images/' . $type;
        if (!empty($params)) {
            $url .= '?' . http_build_query($params);
        }
        return $url;
    }

    /**
     * Gets the URL for a primary image (poster).
     *
     * @param string $item_id
     *   The item ID.
     * @param int $max_width
     *   Optional maximum width.
     *
     * @return string
     *   The image URL.
     */
    public function getPosterUrl($item_id, $max_width = 300)
    {
        return $this->getImageUrl($item_id, 'Primary', ['maxWidth' => $max_width]);
    }

    /**
     * Gets the URL for a backdrop image.
     *
     * @param string $item_id
     *   The item ID.
     * @param int $max_width
     *   Optional maximum width.
     *
     * @return string
     *   The image URL.
     */
    public function getBackdropUrl($item_id, $max_width = 1280)
    {
        return $this->getImageUrl($item_id, 'Backdrop', ['maxWidth' => $max_width]);
    }

    /**
     * Gets the URL for a thumbnail image.
     *
     * @param string $item_id
     *   The item ID.
     * @param int $max_width
     *   Optional maximum width.
     *
     * @return string
     *   The image URL.
     */
    public function getThumbnailUrl($item_id, $max_width = 200)
    {
        return $this->getImageUrl($item_id, 'Thumb', ['maxWidth' => $max_width]);
    }

    // =========================================================================
    // PLAYBACK OPERATIONS
    // =========================================================================

    /**
     * Gets playback info for an item.
     *
     * @param string $item_id
     *   The item ID.
     * @param string $user_id
     *   The user ID.
     *
     * @return array|null
     *   Playback info or NULL on failure.
     */
    public function getPlaybackInfo($item_id, $user_id)
    {
        return $this->request('GET', '/Items/' . $item_id . '/PlaybackInfo', [
            'query' => ['userId' => $user_id],
        ]);
    }

    /**
     * Gets the stream URL for an item.
     *
     * @param string $item_id
     *   The item ID.
     * @param string $container
     *   The container format (mp4, mkv, etc.).
     *
     * @return string
     *   The stream URL.
     */
    public function getStreamUrl($item_id, $container = 'mp4')
    {
        return $this->getServerUrl() . '/Videos/' . $item_id . '/stream.' . $container . '?static=true&api_key=' . $this->getApiKey();
    }

}
