<?php

namespace Drupal\json_field_query\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\json_field_query\Service\JsonQueryService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'json_query_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "json_query_formatter",
 *   label = @Translation("JSON Query"),
 *   field_types = {
 *     "json_query"
 *   }
 * )
 */
class JsonQueryFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The JSON query service.
   *
   * @var \Drupal\json_field_query\Service\JsonQueryService
   */
  protected $jsonQueryService;

  /**
   * Constructs a JsonQueryFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\json_field_query\Service\JsonQueryService $json_query_service
   *   The JSON query service.
   */
  public function __construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings, JsonQueryService $json_query_service) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->jsonQueryService = $json_query_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('json_field_query.json_query_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'json_query' => '',
      'display_type' => 'string',
      'prefix' => '',
      'suffix' => '',
      'fallback' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = [];

    $element['json_query'] = [
      '#type' => 'textfield',
      '#title' => $this->t('JSON Query Path'),
      '#description' => $this->t('Use dot notation to query the JSON. Example: <code>user.name</code> or <code>products.0.price</code>'),
      '#default_value' => $this->getSetting('json_query'),
    ];

    $element['display_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Display Type'),
      '#options' => [
        'string' => $this->t('String'),
        'number' => $this->t('Number'),
        'boolean' => $this->t('Boolean'),
        'json' => $this->t('Formatted JSON'),
      ],
      '#default_value' => $this->getSetting('display_type'),
      '#description' => $this->t('How to format the extracted value.'),
    ];

    $element['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix'),
      '#default_value' => $this->getSetting('prefix'),
      '#description' => $this->t('Text to display before the value.'),
    ];

    $element['suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Suffix'),
      '#default_value' => $this->getSetting('suffix'),
      '#description' => $this->t('Text to display after the value.'),
    ];

    $element['fallback'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Fallback Value'),
      '#default_value' => $this->getSetting('fallback'),
      '#description' => $this->t('Display this text if the query returns no result.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $query = $this->getSetting('json_query');
    if (!empty($query)) {
      $summary[] = $this->t('Query: @query', ['@query' => $query]);
      $summary[] = $this->t('Display as: @type', ['@type' => $this->getSetting('display_type')]);
      
      if (!empty($this->getSetting('prefix'))) {
        $summary[] = $this->t('Prefix: @prefix', ['@prefix' => $this->getSetting('prefix')]);
      }
      if (!empty($this->getSetting('suffix'))) {
        $summary[] = $this->t('Suffix: @suffix', ['@suffix' => $this->getSetting('suffix')]);
      }
      if (!empty($this->getSetting('fallback'))) {
        $summary[] = $this->t('Fallback: @fallback', ['@fallback' => $this->getSetting('fallback')]);
      }
    } else {
      $summary[] = $this->t('No query configured');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $element = [];
    $settings = $this->getSettings();

    foreach ($items as $delta => $item) {
      $value = $this->jsonQueryService->queryJson($item->value, $settings['json_query']);
      
      if ($value === NULL && !empty($settings['fallback'])) {
        $value = $settings['fallback'];
      }

      if ($value !== NULL) {
        $display_value = $this->formatValue($value, $settings['display_type']);
        
        $element[$delta] = [
          '#markup' => $settings['prefix'] . $display_value . $settings['suffix'],
        ];
      } else {
        $element[$delta] = [];
      }
    }

    return $element;
  }

  /**
   * Format the value based on display type.
   *
   * @param mixed $value
   *   The value to format.
   * @param string $display_type
   *   The display type.
   *
   * @return string
   *   The formatted value.
   */
  protected function formatValue($value, $display_type) {
    switch ($display_type) {
      case 'number':
        return is_numeric($value) ? (string) $value : (string) floatval($value);

      case 'boolean':
        return $value ? $this->t('Yes') : $this->t('No');

      case 'json':
        return '<pre>' . json_encode($value, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . '</pre>';

      case 'string':
      default:
        if (is_array($value)) {
          return implode(', ', $value);
        }
        if (is_bool($value)) {
          return $value ? $this->t('True') : $this->t('False');
        }
        return (string) $value;
    }
  }

}
