<?php

namespace Drupal\json_field_query\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'json_query_textarea' widget.
 *
 * @FieldWidget(
 *   id = "json_query_textarea",
 *   label = @Translation("JSON Textarea"),
 *   field_types = {
 *     "json_query"
 *   }
 * )
 */
class JsonQueryWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'rows' => 10,
      'placeholder' => '{"key": "value"}',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = [];

    $element['rows'] = [
      '#type' => 'number',
      '#title' => $this->t('Rows'),
      '#default_value' => $this->getSetting('rows'),
      '#min' => 1,
      '#max' => 50,
    ];

    $element['placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder'),
      '#default_value' => $this->getSetting('placeholder'),
      '#description' => $this->t('Text that will be shown inside the field until a value is entered.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $summary[] = $this->t('Rows: @rows', ['@rows' => $this->getSetting('rows')]);
    if (!empty($this->getSetting('placeholder'))) {
      $summary[] = $this->t('Placeholder: @placeholder', ['@placeholder' => $this->getSetting('placeholder')]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element['value'] = $element + [
      '#type' => 'textarea',
      '#default_value' => isset($items[$delta]->value) ? $items[$delta]->value : NULL,
      '#rows' => $this->getSetting('rows'),
      '#placeholder' => $this->getSetting('placeholder'),
      '#attributes' => [
        'class' => ['json-field-textarea'],
      ],
      '#element_validate' => [
        [$this, 'validateJson'],
      ],
    ];

    return $element;
  }

  /**
   * Validate the JSON input.
   */
  public function validateJson($element, FormStateInterface $form_state, $form) {
    $value = $element['#value'];
    
    if (!empty($value) && $this->getFieldSetting('validate_json')) {
      json_decode($value);
      if (json_last_error() !== JSON_ERROR_NONE) {
        $form_state->setError($element, $this->t('The JSON input is not valid. Error: @error', [
          '@error' => json_last_error_msg(),
        ]));
      }
    }
  }

}
