<?php

namespace Drupal\json_field_query\Service;

/**
 * Service for JSON query operations.
 */
class JsonQueryService {

  /**
   * Query JSON data using dot notation.
   *
   * @param string $json
   *   The JSON string to query.
   * @param string $query
   *   The query path in dot notation (e.g., 'user.name' or 'products.0.price').
   *
   * @return mixed
   *   The queried value or NULL if not found.
   */
  public function queryJson($json, $query) {
    if (empty($json) || empty($query)) {
      return NULL;
    }

    try {
      $data = json_decode($json, TRUE);
      
      if (json_last_error() !== JSON_ERROR_NONE) {
        return NULL;
      }

      $path = explode('.', $query);
      $result = $data;

      foreach ($path as $key) {
        // Handle array indices
        if (is_numeric($key)) {
          $key = (int) $key;
        }

        if (!isset($result[$key])) {
          return NULL;
        }
        $result = $result[$key];
      }

      return $result;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Get all available paths from JSON for autocomplete.
   *
   * @param string $json
   *   The JSON string to analyze.
   * @param string $prefix
   *   Current path prefix for recursion.
   *
   * @return array
   *   Array of available paths.
   */
  public function getJsonPaths($json, $prefix = '') {
    $paths = [];
    
    if (empty($json)) {
      return $paths;
    }

    try {
      $data = json_decode($json, TRUE);
      
      if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
        return $paths;
      }

      foreach ($data as $key => $value) {
        $current_path = $prefix ? $prefix . '.' . $key : $key;
        
        if (is_array($value)) {
          $paths[] = $current_path;
          $paths = array_merge($paths, $this->getJsonPaths(json_encode($value), $current_path));
        } else {
          $paths[] = $current_path;
        }
      }
    }
    catch (\Exception $e) {
      // Silently fail for path generation
    }

    return $paths;
  }

}
