/**
 * @file
 * JSON Viewer module JavaScript with search functionality.
 */

(function ($, Drupal) {
  "use strict";

  /**
   * Provides a JSON viewer functionality with controls.
   */
  Drupal.behaviors.jsonPreViewer = {
    attach: function (context, settings) {
      once("jsonPreViewer", ".json-viewer-main-container", context).forEach(
        function (viewerElement) {
          const $jsonViewerContainer = $(viewerElement);
          const initialLevel =
            parseInt(
              $jsonViewerContainer
                .find(".json-container")
                .attr("data-expansion-level")
            ) || 1;

          function jsonToHtmlView(key, value, depth = 0) {
            let html = "";
            let type = Array.isArray(value) ? "array" : typeof value;

            if (type === "object" || type === "array") {
              const isEmpty = Object.keys(value).length === 0;
              const toggleButton = isEmpty
                ? ""
                : '<span class="toggle-button">+</span>';
              const nodeClass = type === "array" ? "array-node" : "object-node";

              html +=
                '<div class="json-node ' +
                nodeClass +
                '" data-depth="' +
                depth +
                '">';
              html +=
                '<span class="json-key">' +
                toggleButton +
                " " +
                key +
                ":</span> ";
              html +=
                '<span class="json-value">' +
                (type === "array"
                  ? "[" + value.length + "]"
                  : "{" + Object.keys(value).length + "}") +
                "</span>";

              if (!isEmpty) {
                html += '<div class="json-children" style="display: none;">';
                for (const childKey in value) {
                  html += jsonToHtmlView(childKey, value[childKey], depth + 1);
                }
                html += "</div>";
              }

              html += "</div>";
            } else {
              let formattedValue = value;
              if (typeof value === "string") {
                formattedValue = '"' + value + '"';
              }
              html += '<div class="json-node" data-depth="' + depth + '">';
              html += '<span class="json-key">' + key + ":</span> ";
              html += '<span class="json-value">' + formattedValue + "</span>";
              html += "</div>";
            }

            return html;
          }

          function expandToLevelViewer(level) {
            $jsonViewerContainer.find(".json-node").each(function () {
              const $node = $(this);
              const depth = parseInt($node.attr("data-depth"));
              const $children = $node.find("> .json-children");
              const $toggleButton = $node.find("> .json-key .toggle-button");

              if ($toggleButton.length) {
                if (depth < level) {
                  $children.show();
                  $toggleButton.text("-").addClass("expanded");
                } else {
                  $children.hide();
                  $toggleButton.text("+").removeClass("expanded");
                }
              }
            });
          }

          function renderJsonTreeViewer(jsonData) {
            $jsonViewerContainer.find(".json-tree").empty();
            $jsonViewerContainer
              .find(".json-tree")
              .append(jsonToHtmlView("data", jsonData));
            expandToLevelViewer(initialLevel);
          }

          function updateLineNumbersViewer() {
            const $textarea = $jsonViewerContainer.find(".json-input");
            const lines = $textarea.val().split("\n").length;
            const $lineNumbersDiv = $jsonViewerContainer.find(".line-numbers");
            let lineNumbersHtml = "";

            for (let i = 1; i <= lines; i++) {
              lineNumbersHtml += '<span class="line">' + i + "</span>";
            }

            $lineNumbersDiv.html(lineNumbersHtml);
          }

          function searchJsonTree(query) {
            if (!query) {
              $jsonViewerContainer
                .find(".json-node")
                .removeClass("search-match");
              return;
            }

            const matchedNodes = [];

            // Search for matching keys or values.
            $jsonViewerContainer.find(".json-node").each(function () {
              const $node = $(this);
              const keyText = $node.find("> .json-key").text().toLowerCase();
              const valueText = $node
                .find("> .json-value")
                .text()
                .toLowerCase();

              if (keyText.includes(query) || valueText.includes(query)) {
                matchedNodes.push($node);
              }
            });

            if (!matchedNodes.length) {
              alert("No matches found.");
              return;
            }

            // Clear previous matches.
            $jsonViewerContainer.find(".json-node").removeClass("search-match");

            // Highlight and expand matched nodes.
            matchedNodes.forEach(($node) => {
              $node.addClass("search-match");

              // Expand all parents of matched node.
              let current = $node;
              while (current.length && !current.is(".json-tree")) {
                const $childrenContainer = current.find("> .json-children");
                const $toggleButton = current.find(
                  "> .json-key .toggle-button"
                );

                if ($toggleButton.length) {
                  $childrenContainer.show();
                  $toggleButton.text("-").addClass("expanded");
                }

                current = current.parent().closest(".json-node");
              }
            });
          }

          // Event Handlers
          $jsonViewerContainer
            .find(".view-json-button")
            .on("click", function () {
              const input = $jsonViewerContainer.find(".json-input").val();
              try {
                const jsonData = JSON.parse(input);
                renderJsonTreeViewer(jsonData);
              } catch (err) {
                alert("Invalid JSON:\n" + err.message);
              }
            });

          $jsonViewerContainer
            .find(".expansion-level-input")
            .on("change keyup", function () {
              const level = parseInt($(this).val());
              if (!isNaN(level)) {
                expandToLevelViewer(level);
              }
            });

          $jsonViewerContainer
            .find(".expand-all-button")
            .on("click", function () {
              expandToLevelViewer(999);
              $jsonViewerContainer.find(".expansion-level-input").val(999);
            });

          $jsonViewerContainer
            .find(".collapse-all-button")
            .on("click", function () {
              expandToLevelViewer(0);
              $jsonViewerContainer.find(".expansion-level-input").val(0);
            });

          $jsonViewerContainer
            .find(".fullscreen-toggle-button")
            .on("click", function () {
              const container = $jsonViewerContainer.find(".json-container")[0];
              if (document.fullscreenElement) {
                document.exitFullscreen();
              } else {
                container.requestFullscreen().catch((err) => {
                  alert(
                    "Error attempting to enable full-screen mode: " +
                      err.message
                  );
                });
              }
            });

          $(document).on("fullscreenchange", function () {
            const button = $jsonViewerContainer.find(
              ".fullscreen-toggle-button"
            );
            if (document.fullscreenElement) {
              button.text("Exit Fullscreen");
            } else {
              button.text("Fullscreen");
            }
          });

          $jsonViewerContainer.find(".json-input").on("scroll", function () {
            $jsonViewerContainer
              .find(".line-numbers")
              .scrollTop($(this).scrollTop());
          });

          $jsonViewerContainer
            .find(".json-input")
            .on("input propertychange", function () {
              updateLineNumbersViewer();
            });

          // Search input handlers
          $jsonViewerContainer
            .find(".json-search-button")
            .on("click", function () {
              const query = $jsonViewerContainer
                .find(".json-search-input")
                .val()
                .toLowerCase();
              searchJsonTree(query);
            });

          $jsonViewerContainer
            .find(".json-search-input")
            .on("keyup", function (e) {
              if (e.key === "Enter") {
                const query = $(this).val().toLowerCase();
                searchJsonTree(query);
              }
            });

          // Initial setup on load
          updateLineNumbersViewer();
          try {
            $(document).ready(function () {
              const input = $jsonViewerContainer.find(".json-input").val();
              const jsonData = JSON.parse(input);
              renderJsonTreeViewer(jsonData);
            });
          } catch (e) {
            console.log("e", e);
            $jsonViewerContainer
              .find(".json-tree")
              .html(
                '<div class="json-viewer-error">Invalid JSON format.</div>'
              );
          }
        }
      );
    },
  };
})(jQuery, Drupal);
