<?php

namespace Drupal\json_viewer\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'JSON Previewer' block.
 *
 * @Block(
 * id = "json_previewer_block",
 * admin_label = @Translation("JSON Previewer"),
 * category = @Translation("Custom")
 * )
 */
class JsonPreviewerBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'json_data' => "{
          'user': {
            'name': 'John Doe',
            'age': 30,
            'hobbies': ['reading', 'gaming']
          }
        }",
      'color_palette' => 'color-palette-default',
      'expansion_level' => 1,
      'show_expand_all' => TRUE,
      'show_collapse_all' => TRUE,
      'show_expansion_level' => TRUE,
      'show_search' => TRUE,
      'show_fullscreen' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $config = $this->getConfiguration();
    $json_viewer_palettes = json_viewer_get_color_palettes();
    $options = ['color-palette-default' => $this->t('Default')] + $json_viewer_palettes;

    $form['json_data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Initial JSON Data'),
      '#default_value' => $config['json_data'],
      '#description' => $this->t('Enter the JSON data to be displayed in the viewer.'),
    ];

    $form['color_palette'] = [
      '#type' => 'select',
      '#title' => $this->t('Color Palette'),
      '#options' => $options,
      '#default_value' => $config['color_palette'],
      '#description' => $this->t('Choose a color palette for the JSON viewer.'),
    ];

    // Fieldset for control button checkboxes.
    $form['controls'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Display Controls'),
      '#tree' => TRUE,
    ];
    $form['controls']['show_expand_all'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show "Expand All" button'),
      '#default_value' => $config['show_expand_all'],
    ];
    $form['controls']['show_collapse_all'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show "Collapse All" button'),
      '#default_value' => $config['show_collapse_all'],
    ];
    $form['controls']['show_expansion_level'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show expansion level input'),
      '#default_value' => $config['show_expansion_level'],
    ];
    $form['controls']['show_fullscreen'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show "Fullscreen" button'),
      '#default_value' => $config['show_fullscreen'],
    ];
    $form['controls']['show_search'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show "Search" input'),
      '#default_value' => $config['show_search'],
    ];

    $form['expansion_level'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Expansion Level'),
      '#default_value' => $config['expansion_level'],
      '#min' => 0,
      '#description' => $this->t('Set the number of levels to expand by default.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockValidate($form, FormStateInterface $form_state) {
    $json_data = $form_state->getValue('json_data');

    // Validate JSON.
    json_decode($json_data);
    if (json_last_error() !== JSON_ERROR_NONE) {
      $form_state->setErrorByName('json_data', $this->t('Invalid JSON data. Please fix the errors.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    $this->configuration['json_data'] = $form_state->getValue('json_data');
    $this->configuration['color_palette'] = $form_state->getValue('color_palette');
    $this->configuration['expansion_level'] = $form_state->getValue('expansion_level');
    $this->configuration['show_expand_all'] = $form_state->getValue(['controls', 'show_expand_all']);
    $this->configuration['show_collapse_all'] = $form_state->getValue(['controls', 'show_collapse_all']);
    $this->configuration['show_expansion_level'] = $form_state->getValue(['controls', 'show_expansion_level']);
    $this->configuration['show_search'] = $form_state->getValue(['controls', 'show_search']);
    $this->configuration['show_fullscreen'] = $form_state->getValue(['controls', 'show_fullscreen']);
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $config = $this->getConfiguration();

    return [
      '#theme' => 'json_previewer_block',
      '#json_data' => $config['json_data'],
      '#color_palette' => $config['color_palette'],
      '#expansion_level' => $config['expansion_level'],
      '#show_expand_all' => $config['show_expand_all'],
      '#show_collapse_all' => $config['show_collapse_all'],
      '#show_expansion_level' => $config['show_expansion_level'],
      '#show_search' => $config['show_search'],
      '#show_fullscreen' => $config['show_fullscreen'],
      '#attached' => [
        'library' => ['json_viewer/json_previewer'],
      ],
      '#cache' => [
        'max-age' => 0,
      ],
    ];
  }

}
