<?php

namespace Drupal\json_viewer\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\media\Entity\Media;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Plugin implementation of the 'json_media_viewer' formatter.
 *
 * @FieldFormatter(
 * id = "json_media_viewer",
 * label = @Translation("JSON Media Viewer"),
 * field_types = {
 * "entity_reference"
 * }
 * )
 */
class JsonMediaViewerFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a JsonMediaViewerFormatter object.
   *
   * @param string $plugin_id
   *   The plugin ID for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field.
   * @param array $settings
   *   The settings array.
   * @param string $label
   *   The formatter label.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Third party settings.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    FileSystemInterface $file_system,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('file_system')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'color_palette' => 'color-palette-default',
      'level' => '0',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);

    $json_viewer_palettes = json_viewer_get_color_palettes();
    $options = ['color-palette-default' => $this->t('Default')] + $json_viewer_palettes;

    $form['color_palette'] = [
      '#type' => 'select',
      '#title' => $this->t('Color Palette'),
      '#options' => $options,
      '#default_value' => $this->getSetting('color_palette'),
      '#description' => $this->t('Choose a color palette for the JSON viewer.'),
    ];
    $form['level'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Expansion Level'),
      '#default_value' => $this->getSetting('level') ?? 0,
      '#description' => $this->t('Enter Default Expansion Level for the JSON viewer.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $palette = $this->getSetting('color_palette');
    $level = $this->getSetting('level');
    $palettes = ['color-palette-default' => $this->t('Default')] + json_viewer_get_color_palettes();
    $summary[] = $this->t('Color Palette: @palette', ['@palette' => $palettes[$palette] ?? $this->t('Default')]);
    $summary[] = $this->t('Default Expansion Level: @level', ['@level' => $level ?? 0]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $selected_palette = $this->getSetting('color_palette');
    $level = $this->getSetting('level');

    foreach ($items as $delta => $item) {
      $media = $item->get('entity')->getTarget()->getValue();

      // Ensure the referenced entity is a Media entity.
      if ($media instanceof Media) {
        $source = $media->getSource();
        $source_field_name = $source->getSourceFieldDefinition($media->bundle->entity)->getName();
        $file = $media->get($source_field_name)->entity;

        // Check if the source field contains a file entity.
        if ($file && $file->getMimeType() === 'application/json') {
          $file_path = $this->fileSystem->realpath($file->getFileUri());

          if (file_exists($file_path)) {
            $json_data = file_get_contents($file_path);
            try {
              $decoded_json = json_decode($json_data, FALSE, 512, JSON_THROW_ON_ERROR);
              $formatted_json = json_encode($decoded_json, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

              $elements[$delta] = [
                '#theme' => 'json_viewer',
                '#json' => $formatted_json,
                '#color_palette' => $selected_palette,
                '#level' => $level,
                '#attached' => [
                  'library' => [
                    'json_viewer/json_viewer',
                  ],
                ],
              ];
            }
            catch (\JsonException $e) {
              $elements[$delta] = [
                '#markup' => '<div class="json-viewer-error">Invalid JSON file content: ' . htmlspecialchars($json_data) . '</div>',
              ];
            }
          }
          else {
            $elements[$delta] = [
              '#markup' => '<div class="json-viewer-error">File not found.</div>',
            ];
          }
        }
        else {
          $elements[$delta] = [
            '#markup' => '<div class="json-viewer-error">The referenced media entity does not contain a JSON file.</div>',
          ];
        }
      }
    }
    return $elements;
  }

}
