/**
 * @file
 * JSON Viewer module JavaScript.
 */

(function ($, Drupal) {
  "use strict";

  /**
   * Provides a JSON viewer functionality.
   */
  Drupal.behaviors.jsonViewer = {
    attach: function (context, settings) {
      // Use .once() to ensure the behavior is applied only once per element.
      // The selector here should be the container for your JSON viewer.
      once("jsonViewer", ".json-container", context).forEach(function (
        viewerElement,
      ) {
        const $viewerContainer = $(viewerElement);
        const jsonDataInput = $viewerContainer.find(".json-data").val();
        let jsonData;

        try {
          // Attempt to parse the JSON.
          jsonData = JSON.parse(jsonDataInput);

          function expandToLevel(level) {
            $viewerContainer.find(".json-node").each(function () {
              const $node = $(this);
              const depth = parseInt($node.attr("data-depth"));
              const $children = $node.find("> .json-children");
              const $toggleButton = $node.find("> .json-key .toggle-button");

              if (depth < level) {
                $children.show();
                $toggleButton.text("-");
              } else {
                $children.hide();
                $toggleButton.text("+");
              }
            });
          }

          /**
           * Recursively converts a JSON object to an expandable/collapsible HTML tree.
           * @param {string} key - The key of the current JSON node.
           * @param {*} value - The value of the current JSON node.
           * @param {number} depth - The current depth of the JSON node.
           * @returns {string} The HTML string for the JSON node.
           */
          const jsonToHtml = function (key, value, depth) {
            let html = "";
            const type = Array.isArray(value) ? "array" : typeof value;

            if (type === "object" || type === "array") {
              const isEmpty = Object.keys(value).length === 0;
              const toggleButton = isEmpty
                ? ""
                : '<span class="toggle-button">+</span>';
              const nodeClass = type === "array" ? "array-node" : "object-node";

              // Add the data-depth attribute to the node.
              html +=
                '<div class="json-node ' +
                nodeClass +
                '" data-depth="' +
                depth +
                '">';
              html +=
                '<span class="json-key">' +
                toggleButton +
                " " +
                key +
                ":</span> ";
              html +=
                '<span class="json-value">' +
                (type === "array" ? "[...]" : "{...}") +
                "</span>";

              if (!isEmpty) {
                html += '<div class="json-children" style="display: none;">';
                for (const childKey in value) {
                  // Recursively call with incremented depth.
                  html += jsonToHtml(childKey, value[childKey], depth + 1);
                }
                html += "</div>";
              }

              html += "</div>";
            } else {
              // Handle primitive types.
              let formattedValue = value;
              if (typeof value === "string") {
                formattedValue = '"' + value + '"';
              }
              // Add the data-depth attribute to the node.
              html += '<div class="json-node" data-depth="' + depth + '">';
              html += '<span class="json-key">' + key + ":</span> ";
              html += '<span class="json-value">' + formattedValue + "</span>";
              html += "</div>";
            }

            return html;
          };

          // Clear previous content and append the new tree.
          // Start the initial call with a depth of 0.
          $viewerContainer
            .find(".json-tree")
            .empty()
            .append(jsonToHtml("data", jsonData, 0));

          const level = parseInt($viewerContainer.attr("data-level"));

          // Now this function will correctly expand the nodes.
          expandToLevel(level);
        } catch (e) {
          // If JSON is invalid, display an error message.
          $viewerContainer
            .find(".json-tree")
            .html('<div class="json-viewer-error">Invalid JSON format.</div>');
        }

        // Handle the click event for toggling the tree nodes.
        $viewerContainer.on("click", ".toggle-button", function () {
          const button = $(this);
          const children = button
            .closest(".json-node")
            .find("> .json-children");

          children.slideToggle("fast", function () {
            if (children.is(":visible")) {
              button.text("-");
            } else {
              button.text("+");
            }
          });
        });
      });
    },
  };
})(jQuery, Drupal);
