<?php

namespace Drupal\json_viewer\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'json_viewer' formatter.
 *
 * @FieldFormatter(
 * id = "json_viewer",
 * label = @Translation("JSON Viewer"),
 * field_types = {
 * "string",
 * "string_long",
 * "text",
 * "text_long",
 * "text_with_summary"
 * }
 * )
 */
class JsonViewerFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
    // Set a default value.
      'color_palette' => 'color-palette-default',
      'level' => '0',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);

    // This function must be defined in your .module file.
    $json_viewer_palettes = json_viewer_get_color_palettes();
    $options = ['color-palette-default' => $this->t('Default')] + $json_viewer_palettes;

    $form['color_palette'] = [
      '#type' => 'select',
      '#title' => $this->t('Color Palette'),
      '#options' => $options,
      '#default_value' => $this->getSetting('color_palette'),
      '#description' => $this->t('Choose a color palette for the JSON viewer.'),
    ];
    $form['level'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Expansion Level'),
      '#default_value' => $this->getSetting('level') ?? 0,
      '#description' => $this->t('Enter Default Expansion Level for the JSON viewer.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $palette = $this->getSetting('color_palette');
    $level = $this->getSetting('level');
    $palettes = ['color-palette-default' => $this->t('Default')] + json_viewer_get_color_palettes();
    $summary[] = $this->t('Color Palette: @palette', ['@palette' => $palettes[$palette] ?? $this->t('Default')]);
    $summary[] = $this->t('Default Expansion Level: @level', ['@level' => $level ?? 0]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $selected_palette = $this->getSetting('color_palette');
    $level = $this->getSetting('level');

    foreach ($items as $delta => $item) {
      $value = $item->value;
      if (!empty($value)) {
        try {
          $decoded_json = json_decode($value, FALSE, 512, JSON_THROW_ON_ERROR);
          $formatted_json = json_encode($decoded_json, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

          $elements[$delta] = [
            '#theme' => 'json_viewer',
            '#json' => $formatted_json,
            '#color_palette' => $selected_palette,
            '#level' => $level,
            '#attached' => [
              'library' => [
                'json_viewer/json_viewer',
              ],
            ],
          ];
        }
        catch (\JsonException $e) {
          $elements[$delta] = [
            '#markup' => '<div class="json-viewer-error">Invalid JSON: ' . htmlspecialchars($value) . '</div>',
          ];
        }
      }
    }
    return $elements;
  }

}
