import { useState, useEffect } from 'react';
import axios from 'axios';

export const useEntityTypes = (baseUrl) => {
  const [entities, setEntities] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    // Fetch JSON:API resources on component mount
    fetchEntities();
  }, []);

  const fetchEntities = async () => {
    setLoading(true);
    setError(null);

    try {
      // Fetch OpenAPI JSON:API specification from the OpenAPI endpoint
      // Get the base path from baseUrl to handle custom JSON:API paths
      const openApiUrl = drupalSettings.path.baseUrl + "openapi/jsonapi"
      console.log('Fetching OpenAPI schema from:', openApiUrl);
      
      const response = await axios.get(openApiUrl);
      console.log('OpenAPI response status:', response.status);
      
      if (response.data && response.data.paths) {
        console.log('OpenAPI response has paths property');
        const entityTypes = extractEntityTypesFromOpenApi(response.data);
        console.log('Setting entities state with:', entityTypes.length, 'entity types');
        setEntities(entityTypes);
      } else {
        console.error('Invalid OpenAPI response format - missing paths property');
        if (response.data) {
          console.log('Response data keys:', Object.keys(response.data));
        }
        setError('Invalid OpenAPI response format');
      }
    } catch (err) {
      console.error('Error fetching OpenAPI resources:', err);
      setError(`Error fetching OpenAPI resources: ${err.message}`);
    } finally {
      setLoading(false);
    }
  };

  const extractEntityTypesFromOpenApi = (data) => {
    const entityTypes = [];
    const entityTypesMap = new Map();
    
    console.log('Extracting entity types from OpenAPI schema');
    console.log('baseUrl:', baseUrl);

    // Extract entity types from OpenAPI paths
    if (data && data.paths) {
      console.log('Number of paths in schema:', Object.keys(data.paths).length);
      
      // Debug: Log the first 5 paths to see their structure
      const pathSample = Object.keys(data.paths).slice(0, 10);
      console.log('Sample paths:', pathSample);
      
      // Get more detailed info about paths structure
      if (pathSample.length > 0) {
        console.log('First path data structure:', JSON.stringify(data.paths[pathSample[0]], null, 2).substring(0, 500));
      }
      
      // Inspect schema structure to understand format
      console.log('Schema structure:', Object.keys(data));
      if (data.components) {
        console.log('Components structure:', Object.keys(data.components));
        if (data.components.schemas) {
          console.log('Number of schemas:', Object.keys(data.components.schemas).length);
          console.log('Schema examples:', Object.keys(data.components.schemas).slice(0, 5));
        }
      }
      
      // Try a more aggressive approach - parse all paths containing /jsonapi/
      // and extract entity type and bundle information
      console.log('Attempting more aggressive path matching approach');
      
      const allJsonapiPaths = Object.keys(data.paths).filter(path => 
        path.includes('/jsonapi/') && 
        // At least has /jsonapi/something/something format
        path.match(/\/jsonapi\/[^/]+\/[^/]+/)
      );
      
      console.log('Found JSON:API paths:', allJsonapiPaths.length);
      
      // Loop through all paths in the OpenAPI specification
      let matchCount = 0;
      Object.keys(data.paths).forEach(path => {
        // Extract entity type and bundle from any path that resembles a JSON:API endpoint
        // This broader pattern captures more path formats
        const matches = path.match(/\/jsonapi\/([^/]+)\/([^/]+)/);
        
        if (matches && matches.length === 3) {
          matchCount++;
          const [_, entityType, bundle] = matches;
          
          // Clean up bundle name - remove trailing parts after /
          // For example, convert "article/{entity}" to just "article"
          const cleanBundle = bundle.split('/')[0];
          
          // Generate a reference URL for this resource
          const resourceUrl = `${baseUrl.replace(/\/$/, '')}/${entityType}/${cleanBundle}`;
          
          console.log('Found match:', path, 'entityType:', entityType, 'bundle:', cleanBundle);
          
          // Check if entity type already exists
          if (!entityTypesMap.has(entityType)) {
            entityTypesMap.set(entityType, {
              name: entityType,
              bundles: []
            });
          }
          
          // Add bundle if not already added
          const existingType = entityTypesMap.get(entityType);
          if (!existingType.bundles.some(b => b.name === cleanBundle)) {
            existingType.bundles.push({
              name: cleanBundle,
              href: resourceUrl
            });
          }
        } else if (path.includes('/jsonapi/')) {
          console.log('No match for path:', path);
        }
      });
      
      // Try alternative approach - look in schema components if available
      if (entityTypesMap.size === 0 && data.components && data.components.schemas) {
        console.log('Trying alternative approach using schema components');
        
        // Look for schemas that might represent entity types
        Object.keys(data.components.schemas).forEach(schemaName => {
          // Look for patterns like "jsonapi--node--article", "node--article", etc.
          const matches = schemaName.match(/(?:jsonapi--)?([^-]+)--([^-]+)$/);
          
          if (matches && matches.length === 3) {
            const [_, entityType, bundle] = matches;
            
            console.log('Found schema match:', schemaName, 'entityType:', entityType, 'bundle:', bundle);
            
            // Generate a reference URL
            const resourceUrl = `${baseUrl.replace(/\/$/, '')}/${entityType}/${bundle}`;
            
            // Add to entity types map
            if (!entityTypesMap.has(entityType)) {
              entityTypesMap.set(entityType, {
                name: entityType,
                bundles: []
              });
            }
            
            // Add bundle if not already added
            const existingType = entityTypesMap.get(entityType);
            if (!existingType.bundles.some(b => b.name === bundle)) {
              existingType.bundles.push({
                name: bundle,
                href: resourceUrl
              });
            }
          }
        });
      }
      
      console.log('Total matches found:', matchCount);
      console.log('Entity types found:', entityTypesMap.size);
      
      // Fallback to hardcoded common entity types if nothing was found
      if (entityTypesMap.size === 0) {
        console.log('Using fallback common entity types');
        
        // Common Drupal entity types and bundles
        const commonTypes = [
          { type: 'node', bundles: ['article', 'page'] },
          { type: 'user', bundles: ['user'] },
          { type: 'taxonomy_term', bundles: ['tags'] },
          { type: 'file', bundles: ['file'] },
          { type: 'media', bundles: ['image'] }
        ];
        
        commonTypes.forEach(item => {
          const entityType = item.type;
          entityTypesMap.set(entityType, {
            name: entityType,
            bundles: item.bundles.map(bundle => ({
              name: bundle,
              href: `${baseUrl.replace(/\/$/, '')}/${entityType}/${bundle}`
            }))
          });
        });
      }
      
    } else {
      console.error('OpenAPI schema is missing paths property or is invalid');
      console.log('Schema data:', data);
    }
    
    // Convert map to array
    entityTypesMap.forEach(type => {
      entityTypes.push(type);
    });
    
    console.log('Final entity types:', entityTypes.length);
    console.log(entityTypes);

    return entityTypes;
  };

  return {
    entities,
    loading,
    error,
    fetchEntities
  };
};