#!/bin/bash

#
# Unified test script for JSON:API Query Builder
#
# This script replaces all previous test scripts with a single, streamlined approach.
#

# Exit on error
set -e

# Default options
HEADLESS=true
REGENERATE=false
PORT=8888
TEST_DIR="/tmp/drupal_test_jsonapi_query_builder"

# Process arguments
for arg in "$@"; do
  case $arg in
    --headless=*)
      HEADLESS="${arg#*=}"
      shift
      ;;
    --regen=*)
      REGENERATE="${arg#*=}"
      shift
      ;;
    --help)
      echo "Usage: $0 [options]"
      echo "Options:"
      echo "  --headless=true|false   Run tests in headless mode (default: true)"
      echo "  --regen=true|false      Regenerate test environment (default: false)"
      echo "  --help                  Show this help message"
      exit 0
      ;;
    *)
      # Unknown option
      ;;
  esac
done

# Create screenshots directory
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
mkdir -p "$SCRIPT_DIR/test/screenshots"

# Display test configuration
echo "Test configuration:"
echo "- Headless mode: $HEADLESS"
echo "- Regenerate environment: $REGENERATE"

# Check for Chrome processes from previous tests and clean up
echo "Checking for Chrome processes from previous test runs..."
if pgrep -f "chrome.*jsonapi-test" > /dev/null; then
  echo "Killing previous Chrome test processes..."
  pkill -f "chrome.*jsonapi-test" || true
  sleep 2
fi

# Get the absolute path to the module root
MODULE_DIR=$(cd "$(dirname "$0")/../.." && pwd)

# Handle environment regeneration if requested
if [ "$REGENERATE" = "true" ]; then
  echo "Regenerating test environment..."
  
  # Kill any running server
  if nc -z localhost $PORT 2>/dev/null; then
    echo "Stopping server on port $PORT"
    lsof -ti:$PORT | xargs kill -9 2>/dev/null || true
    sleep 1
  fi
  
  # Remove existing test directory
  if [ -d "$TEST_DIR" ]; then
    echo "Removing existing test directory..."
    rm -rf "$TEST_DIR" 2>/dev/null || true
    if [ -d "$TEST_DIR" ]; then
      echo "Warning: Unable to remove directory. You may need to run:"
      echo "sudo rm -rf $TEST_DIR"
      exit 1
    fi
  fi
  
  # Call the CI test script to regenerate the environment
  cd "$MODULE_DIR"
  chmod +x scripts/ci-e2e-test.sh
  
  echo "Setting up new test environment..."
  HEADLESS=$HEADLESS USE_FIXED_DIR=true FORCE_REGENERATE=true scripts/ci-e2e-test.sh
  
  exit 0
fi

# Check for existing test environment
if [ -d "$TEST_DIR" ] && [ -d "$TEST_DIR/web/core" ]; then
  echo "Found existing Drupal environment at $TEST_DIR"
  
  # Check if server is running
  if nc -z localhost $PORT 2>/dev/null; then
    echo "Server already running on port $PORT"
    TEST_URL="http://localhost:$PORT"
  else
    echo "Starting server for existing Drupal environment..."
    (cd "$TEST_DIR/web" && php -S localhost:$PORT >/dev/null 2>&1) &
    # Give the server some time to start
    sleep 3
    TEST_URL="http://localhost:$PORT"
  fi
  
  # Run the tests
  export TEST_URL
  export HEADLESS
  
  echo "Running tests against $TEST_URL"
  npx jest --config=jest.config.js --maxWorkers=1 --verbose
  
else
  echo "No existing Drupal environment found."
  echo "Please run with --regen=true to create a test environment."
  exit 1
fi

echo "Tests completed successfully"