import React, { useState, useEffect } from 'react';

function SortSelector({ selectedEntity, selectedBundle, sorts, onSortsChange, relatedEntityFields, baseUrl }) {
  const [availableFields, setAvailableFields] = useState([]);
  const [newSort, setNewSort] = useState({ field: '', direction: 'asc' });
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [directFields, setDirectFields] = useState([]);
  const [relationshipFields, setRelationshipFields] = useState([]);

  useEffect(() => {
    if (selectedEntity && selectedBundle) {
      fetchFields();
    } else {
      setAvailableFields([]);
      setDirectFields([]);
      setRelationshipFields([]);
    }
  }, [selectedEntity, selectedBundle]);

  // Effect to process related entity fields when they change
  useEffect(() => {
    if (relatedEntityFields && Object.keys(relatedEntityFields).length > 0) {
      // Extract relationship fields for sorting
      const relFields = [];
      
      // Process each relationship's fields
      Object.entries(relatedEntityFields).forEach(([relationship, fields]) => {
        fields.forEach(field => {
          relFields.push(field.name);
        });
      });
      
      // Update relationship fields
      setRelationshipFields(relFields);
      
      // Combine direct and relationship fields
      setAvailableFields([...directFields, ...relFields]);
      
      console.log('Updated sort fields with related entity fields:', relFields);
    } else if (directFields.length > 0) {
      // If no relationship fields, just use direct fields
      setAvailableFields([...directFields]);
      setRelationshipFields([]);
    }
  }, [relatedEntityFields, directFields]);

  const fetchFields = async () => {
    if (!selectedEntity || !selectedBundle) return;
    
    setLoading(true);
    setError(null);
    
    try {
      // Use the OpenAPI endpoint instead of resource/schema
      const openApiUrl = drupalSettings.path.baseUrl + "openapi/jsonapi"

      const response = await fetch(openApiUrl);
      
      if (response.ok) {
        const data = await response.json();
        
        // Extract fields from the schema
        let fieldNames = [];
        
        // Determine if we're dealing with OpenAPI 2.0 or 3.0
        const isOpenApi2 = data && data.swagger && data.swagger.startsWith('2');
        const isOpenApi3 = data && data.openapi && data.openapi.startsWith('3');
        console.log(`Detected OpenAPI version: ${isOpenApi2 ? '2.0 (Swagger)' : isOpenApi3 ? '3.0' : 'unknown'}`);

        // Get the entity type schema from the correct location based on OpenAPI version
        const resourceType = `${selectedEntity}--${selectedBundle}`;
        
        if (isOpenApi2 && data.definitions && data.definitions[resourceType]) {
          // Handle OpenAPI 2.0 structure
          console.log(`Found schema for ${resourceType} in OpenAPI 2.0 definitions`);
          
          const resourceSchema = data.definitions[resourceType];
          
          // In OpenAPI 2.0, attributes are in data.attributes.properties
          if (resourceSchema.properties?.data?.properties?.attributes?.properties) {
            const attributes = resourceSchema.properties.data.properties.attributes.properties;
            console.log(`Found ${Object.keys(attributes).length} fields in attributes for sorting`);
            
            Object.keys(attributes).forEach(field => {
              fieldNames.push(field);
            });
          }
        } 
        else if (isOpenApi3 && data.components && data.components.schemas && data.components.schemas[resourceType]) {
          // Handle OpenAPI 3.0 structure
          console.log(`Found schema for ${resourceType} in OpenAPI 3.0 components.schemas`);
          
          const resourceSchema = data.components.schemas[resourceType];
          
          // In OpenAPI 3.0, attributes are in properties.data.properties.attributes.properties
          if (resourceSchema.properties?.data?.properties?.attributes?.properties) {
            const attributes = resourceSchema.properties.data.properties.attributes.properties;
            console.log(`Found ${Object.keys(attributes).length} fields in attributes for sorting`);
            
            Object.keys(attributes).forEach(field => {
              fieldNames.push(field);
            });
          }
        }
        
        // If no fields found yet and this is OpenAPI 2.0, try other entity types as reference
        if (fieldNames.length === 0 && isOpenApi2 && data.definitions) {
          console.log(`Schema for ${resourceType} not found or has no attributes. Checking other schemas...`);
          
          // Find any entity-like definition that has attributes
          Object.entries(data.definitions).forEach(([defKey, defSchema]) => {
            // Only check entity-like definitions (has the entity--bundle format)
            if (defKey.includes('--') && 
                defSchema.properties?.data?.properties?.attributes?.properties) {
              const attributes = defSchema.properties.data.properties.attributes.properties;
              
              console.log(`Using fields from ${defKey} as reference for sorting`);
              Object.keys(attributes).forEach(field => {
                if (!fieldNames.includes(field)) {
                  fieldNames.push(field);
                }
              });
              
              // If we've found some fields, stop searching
              if (fieldNames.length > 0) {
                return;
              }
            }
          });
        }
        
        // Add basic fields
        const basicFields = ['id', 'uuid', 'created', 'changed'];
        basicFields.forEach(field => {
          if (!fieldNames.includes(field)) {
            fieldNames.push(field);
          }
        });
        
        // Store direct fields separately
        setDirectFields(fieldNames);
        
        // If we have related entity fields, combine them
        if (relationshipFields.length > 0) {
          setAvailableFields([...fieldNames, ...relationshipFields]);
        } else {
          setAvailableFields(fieldNames);
        }
      } else {
        setError(`Schema endpoint failed with status: ${response.status}`);
        
        // Fallback to the mock fields if the schema endpoint fails
        const mockFields = getMockFields(selectedEntity);
        setDirectFields(mockFields);
        setAvailableFields(mockFields);
      }
    } catch (err) {
      // Handle error and set fallback fields
      setError(`Unable to get fields: ${err.message}`);
      
      // Fallback to the mock fields if there's an error
      const mockFields = getMockFields(selectedEntity);
      setDirectFields(mockFields);
      setAvailableFields(mockFields);
    } finally {
      setLoading(false);
    }
  };

  const getMockFields = (entityType) => {
    switch(entityType) {
      case 'node':
        return ['id', 'title', 'created', 'changed', 'status'];
      case 'user':
        return ['id', 'name', 'created', 'access', 'status'];
      case 'taxonomy_term':
        return ['id', 'name', 'weight'];
      default:
        return ['id', 'created', 'changed'];
    }
  };

  const handleAddSort = () => {
    if (newSort.field) {
      onSortsChange([...sorts, { ...newSort }]);
      setNewSort({ field: '', direction: 'asc' });
    }
  };

  const handleRemoveSort = (index) => {
    const updatedSorts = [...sorts];
    updatedSorts.splice(index, 1);
    onSortsChange(updatedSorts);
  };

  const handleToggleSortDirection = (index) => {
    const updatedSorts = [...sorts];
    updatedSorts[index].direction = updatedSorts[index].direction === 'asc' ? 'desc' : 'asc';
    onSortsChange(updatedSorts);
  };
  
  const handleClearAll = () => {
    onSortsChange([]);
  };

  return (
    <div className="sort-selector form-item">
      <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '8px' }}>
        <h4 className="form-item__label" style={{ margin: 0 }}>Sort</h4>
        <button
          className="button button--small"
          onClick={handleClearAll}
          disabled={sorts.length === 0}
        >
          Clear
        </button>
      </div>
      
      {(selectedEntity && selectedBundle) ? (
        <>
          <div className="claro-details">
            <div className="claro-details__wrapper">
              {loading ? (
                <div className="form-item__description">
                  <div className="spinner" style={{ display: 'inline-block', marginRight: '8px' }}></div>
                  Loading fields for sorting...
                </div>
              ) : error ? (
                <div className="messages messages--error">
                  {error}
                </div>
              ) : (
                <>
                  {sorts.length > 0 ? (
                    <>
                      <h5 className="sort-group-title">Selected Sort Criteria</h5>
                      <div className="fields-chips">
                        {sorts.map((sort, index) => (
                          <span 
                            key={index} 
                            className={`gin-chip gin-chip--primary ${sort.field.includes('.') ? 'gin-chip--relationship' : ''}`}
                          >
                            {sort.field} 
                            <span 
                              className="sort-direction-toggle"
                              onClick={() => handleToggleSortDirection(index)}
                              style={{ cursor: 'pointer', margin: '0 4px' }}
                              data-testid="toggle-sort-button"
                            >
                              {sort.direction === 'asc' ? '↑' : '↓'}
                            </span>
                            <button 
                              className="gin-chip--remove"
                              onClick={() => handleRemoveSort(index)}
                              aria-label={`Remove ${sort.field} sort`}
                            >
                              ✕
                            </button>
                          </span>
                        ))}
                      </div>
                    </>
                  ) : (
                    <div className="form-item__description">
                      No sort criteria applied yet
                    </div>
                  )}
                  
                  <h5 className="sort-group-title" style={{ marginTop: '16px' }}>Add Sort Criterion</h5>
                  <div className="sort-selector__inputs-row" style={{ display: 'flex', flexWrap: 'wrap', gap: '8px' }}>
                    <div className="form-item" style={{ flex: '1 1 200px', minWidth: '150px', marginBottom: '8px' }}>
                      <label className="form-item__label" htmlFor="sort-field">Field</label>
                      <select
                        id="sort-field"
                        className="form-element"
                        value={newSort.field}
                        onChange={(e) => setNewSort({ ...newSort, field: e.target.value })}
                        data-testid="sort-field-select"
                      >
                        <option value="">- Select a field -</option>
                        
                        {/* Direct entity fields */}
                        {directFields.length > 0 && (
                          <optgroup label="Direct Fields">
                            {directFields.map(field => (
                              <option key={field} value={field}>{field}</option>
                            ))}
                          </optgroup>
                        )}
                        
                        {/* Relationship fields */}
                        {relationshipFields.length > 0 && (
                          <optgroup label="Relationship Fields">
                            {relationshipFields.map(field => (
                              <option 
                                key={field} 
                                value={field}
                                style={{ color: '#6b84a3' }}
                              >
                                {field}
                              </option>
                            ))}
                          </optgroup>
                        )}
                      </select>
                    </div>
                    
                    <div className="form-item" style={{ flex: '1 1 200px', minWidth: '150px', marginBottom: '8px' }}>
                      <label className="form-item__label" htmlFor="sort-direction">Direction</label>
                      <select
                        id="sort-direction"
                        className="form-element"
                        value={newSort.direction}
                        onChange={(e) => setNewSort({ ...newSort, direction: e.target.value })}
                      >
                        <option value="asc">Ascending</option>
                        <option value="desc">Descending</option>
                      </select>
                    </div>
                  </div>
                  
                  <div style={{ marginTop: '8px', marginBottom: '16px' }}>
                    <button
                      id="sort-add-button"
                      className="button button--small"
                      onClick={handleAddSort}
                      disabled={!newSort.field}
                      data-testid="add-sort-button"
                    >
                      Add
                    </button>
                  </div>
                  
                  {relationshipFields.length > 0 && (
                    <div className="form-item__description" style={{ marginTop: '8px', color: '#6b84a3' }}>
                      <strong>Note:</strong> Include fields (shown in blue) are available because you've added includes to your query.
                    </div>
                  )}
                </>
              )}
            </div>
          </div>
        </>
      ) : (
        <div className="form-item__description">
          Select an entity type and **bundle** to add sort criteria
        </div>
      )}
    </div>
  );
}

export default SortSelector;