<?php

declare(strict_types=1);

namespace Drupal\Tests\jsonapi_query_builder\Functional;

use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the functionality of the JSON:API Query Builder module.
 */
#[Group("jsonapi_query_builder")]
class JsonApiQueryBuilderTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'jsonapi',
    'jsonapi_query_builder',
    'node',
    'field_ui',
    'openapi',
    'openapi_jsonapi',
    'schemata',
    'schemata_json_schema',
  ];

  /**
   * A test user with permission to access the JSON:API Explorer.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A test user without permission to access the JSON:API Explorer.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create test content type and content.
    $this->createContentType(['type' => 'article', 'name' => 'Article']);
    $this->createNode([
      'title' => 'Test Article 1',
      'type' => 'article',
      'status' => 1,
    ]);
    $this->createNode([
      'title' => 'Test Article 2',
      'type' => 'article',
      'status' => 1,
    ]);

    // Flush caches and rebuild router so JSON:API picks up the new content
    // type.
    \Drupal::service('router.builder')->rebuild();
    drupal_flush_all_caches();

    // Create users.
    $this->privilegedUser = $this->drupalCreateUser(['access jsonapi query builder']);
    $this->unprivilegedUser = $this->drupalCreateUser();

    // Ensure JSON:API is enabled for all entities.
    $this->config('jsonapi.settings')
      ->set('read_only', FALSE)
      ->save();
  }

  /**
   * Tests that the module's routes are accessible to users with permission.
   */
  public function testRouteAccess(): void {
    // Test access for privileged user.
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet(Url::fromRoute('jsonapi_query_builder.app'));
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('JSON:API Query Builder');

    // Test for the React app container.
    $this->assertSession()->elementExists('css', '#jsonapi-query-builder-root');

    // Test access denied for unprivileged user.
    $this->drupalLogin($this->unprivilegedUser);
    $this->drupalGet(Url::fromRoute('jsonapi_query_builder.app'));
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that the module correctly loads its libraries.
   */
  public function testLibraryLoading(): void {
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet(Url::fromRoute('jsonapi_query_builder.app'));

    // Test for CSS and JS from the module.
    $this->assertSession()->responseContains('jsonapi_query_builder/js/react-app/build/static/js/');
  }

  /**
   * Tests that JSON:API is correctly configured and accessible.
   */
  public function testJsonApiAccess(): void {
    // Allow anonymous access to the JSON:API.
    Role::load(RoleInterface::ANONYMOUS_ID)
      ->grantPermission('access jsonapi resource list')
      ->grantPermission('access content')
      ->grantPermission('access openapi api docs')
      ->save();

    // Test JSON:API index endpoint.
    $this->drupalGet('/jsonapi');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains('links');

    // Test OpenAPI JSON:API schema support.
    $this->drupalGet('/openapi/jsonapi');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains('properties');
  }

  /**
   * Tests the user permissions work correctly.
   */
  public function testPermissions(): void {
    // Check that the permission is defined.
    $permissions = $this->container->get('user.permissions')
      ->getPermissions();
    $this->assertArrayHasKey('access jsonapi query builder', $permissions);
  }

}
