<?php

declare(strict_types=1);

namespace Drupal\Tests\jsonapi_query_builder\FunctionalJavascript;

use Behat\Mink\Element\NodeElement;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Core\Url;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\Routing\Exception\RouteNotFoundException;

/**
 * Tests the JavaScript functionality of the JSON:API Query Builder.
 */
#[Group("jsonapi_query_builder")]
class JsonApiQueryBuilderJsTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'jsonapi',
    'jsonapi_query_builder',
    'node',
    'field_ui',
    'openapi',
    'openapi_jsonapi',
    'schemata',
    'schemata_json_schema',
  ];

  /**
   * A user with administrative permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create test content type and content.
    $this->createContentType(['type' => 'article', 'name' => 'Article']);

    // Schemata routes that the OpenAPI endpoint depends on don't get generated
    // automatically in this test, so we need to manually rebuild the router.
    // @todo Consider investigating why this is necessary, because it signifies
    //   that something in the tests is working differently than in normal use,
    //   and jiggering with the system could mask other issues.
    \Drupal::service('router.builder')->rebuild();

    $this->assertRouteExists('schemata.node:article');

    // Create test content.
    $this->createNode([
      'title' => 'Test Article 1',
      'type' => 'article',
      'body' => [
        'value' => 'Test body content',
        'format' => 'plain_text',
      ],
      'status' => 1,
    ]);

    $this->createNode([
      'title' => 'Test Article 2',
      'type' => 'article',
      'body' => [
        'value' => 'Another test article',
        'format' => 'plain_text',
      ],
      'status' => 1,
    ]);

    // Create an admin user.
    $this->adminUser = $this->drupalCreateUser([
      'access jsonapi query builder',
      'access content',
      'administer nodes',
      'administer site configuration',
      'administer users',
      'access content overview',
      'access administration pages',
      'access openapi api docs',
    ]);

    // Ensure JSON:API is enabled for all entities.
    $this->config('jsonapi.settings')
      ->set('read_only', FALSE)
      ->save();
  }

  /**
   * Tests that the React application loads and functions correctly.
   */
  public function testReactAppLoads(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet(Url::fromRoute('jsonapi_query_builder.app'));

    $this->assertReactAppInitializes();

    // Verify UI elements are present.
    $this->assertSession()->elementExists('xpath', '//button[contains(@class, "query-panel-tab") and normalize-space(text())="Fields"]');
    $this->assertSession()->elementExists('xpath', '//button[contains(@class, "query-panel-tab") and normalize-space(text())="Filters"]');
    $this->assertSession()->elementExists('xpath', '//button[contains(@class, "query-panel-tab") and normalize-space(text())="Includes"]');
    $this->assertSession()->elementExists('xpath', '//button[contains(@class, "query-panel-tab") and normalize-space(text())="Sort"]');

    // Verify URL display is present.
    $this->assertSession()->elementExists('xpath', '//div[@data-testid="url-display"]');

    // Verify "Execute Query" button is present.
    $this->getExecuteQueryButton();
  }

  /**
   * Tests that entity types and bundles can be selected.
   */
  public function testEntitySelection(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet(Url::fromRoute('jsonapi_query_builder.app'));

    $this->assertReactAppInitializes();

    // Select node entity type.
    $page = $this->getSession()->getPage();
    $entityTypeSelect = $page->find('css', '#entity-type-select');
    $this->assertNotNull($entityTypeSelect, 'Entity type select not found');

    $entityTypeSelect->selectOption('node');

    // Wait for bundle select to appear.
    $bundleSelect = $this->assertSession()->waitForElement('css', '#bundle-select');
    $this->assertNotNull($bundleSelect, 'Bundle select not found after entity type selection');

    // Select article bundle.
    $bundleSelect->selectOption('article');

    // Wait for URL to update.
    $this->assertSession()->waitForText('/jsonapi/node/article');

    // Verify URL contains correct path.
    $this->assertSession()->pageTextContains('/jsonapi/node/article');
  }

  /**
   * Tests that fields can be selected and deselected.
   */
  public function testFieldsSelection(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet(Url::fromRoute('jsonapi_query_builder.app'));

    $this->assertReactAppInitializes();

    // Select node--article.
    $page = $this->getSession()->getPage();
    $page->find('css', '#entity-type-select')->selectOption('node');
    $this->assertSession()->waitForElement('css', '#bundle-select');
    $page->find('css', '#bundle-select')->selectOption('article');

    // Wait for fields to load.
    $this->assertSession()->waitForElement('css', '.available-fields');

    // Find and click the title field checkbox.
    $titleCheckbox = $page->find('css', 'span[aria-label="Title, string"]');
    $this->assertNotNull($titleCheckbox, 'Title field checkbox not found');
    $titleCheckbox->click();

    // Wait for URL to update.
    $this->assertSession()->waitForText('fields[node--article]=title');

    // Verify URL contains fields parameter.
    $this->assertSession()->pageTextContains('fields[node--article]=title');

    // Click the body field checkbox.
    $bodyCheckbox = $page->find('css', 'span[aria-label="Body, object"]');
    $this->assertNotNull($bodyCheckbox, 'Body field checkbox not found');
    $bodyCheckbox->click();

    // Wait for URL to update.
    $this->assertSession()->waitForText('fields[node--article]=title,body');

    // Verify URL contains both fields.
    $this->assertSession()->pageTextContains('fields[node--article]=title,body');

    // Deselect the title field.
    $titleCheckbox->click();

    // Wait for URL to update.
    $this->assertSession()->waitForText('fields[node--article]=body');

    // Verify URL only contains body field.
    $this->assertSession()->pageTextContains('fields[node--article]=body');
  }

  /**
   * Tests that filters can be added and removed.
   */
  public function testFilters(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet(Url::fromRoute('jsonapi_query_builder.app'));

    $this->assertReactAppInitializes();

    // Click the filters tab.
    $this->clickPanelTab('filters');

    $page = $this->getSession()->getPage();

    // Wait for filter form to appear.
    $this->assertSession()->waitForElement('css', '#filter-field');

    // Click add filter button.
    $page->find('css', '#filter-field')->click();

    // Select title field.
    $page->find('css', '#filter-field')->selectOption('title');

    // Select CONTAINS operator.
    $page->find('css', '#filter-operator')->selectOption('CONTAINS');

    // Enter filter value.
    $page->find('css', '#filter-value')->setValue('Test');

    // Click add filter button.
    $page->find('css', '#filter-add-button')->click();

    // Wait for URL to update.
    $this->assertSession()->waitForText('filter[title][condition][value]=Test');

    // Verify URL contains filter parameter.
    $this->assertSession()->pageTextContains('filter[title][value]=Test');
    $this->assertSession()->pageTextContains('filter[title][operator]=CONTAINS');

    // Find and click the remove filter button.
    $page->find('css', 'button[aria-label="Remove title filter"]')->click();

    // Wait for URL to update (filter should be removed).
    $this->assertSession()->waitForElementRemoved('css', 'button[aria-label="Remove title filter"]');

    // Verify URL no longer contains filter parameter.
    $this->assertSession()->pageTextNotContains('filter[title][value]=Test');
    $this->assertSession()->pageTextNotContains('filter[title][operator]=CONTAINS');
  }

  /**
   * Tests that includes can be added and removed.
   */
  public function testIncludes(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet(Url::fromRoute('jsonapi_query_builder.app'));

    $this->assertReactAppInitializes();

    // Select node--article.
    $page = $this->getSession()->getPage();
    $page->find('css', '#entity-type-select')->selectOption('node');
    $this->assertSession()->waitForElement('css', '#bundle-select');
    $page->find('css', '#bundle-select')->selectOption('article');

    // Wait for fields to load.
    $this->assertSession()->waitForElement('css', '.fields-list');

    // Click the includes tab.
    $this->clickPanelTab('includes');

    // Wait for includes panel to appear.
    $this->assertSession()->waitForElement('css', '.include-selector');

    // Wait for include form to appear.
    $this->assertSession()->waitForElement('css', '.include-form');

    // Select uid relationship.
    $page->find('css', 'span[aria-label="Authored by, user--user"]')->click();

    // Click "Execute Query" button.
    $this->getExecuteQueryButton()->click();

    // Wait for URL to update.
    $this->assertSession()->waitForText('include=uid');

    // Verify URL contains include parameter.
    $this->assertSession()->pageTextContains('include=uid');

    // Find and click the remove include button.
    $page->find('css', 'span[aria-label="Authored by, user--user, , Included in response"]	')->click();

    // Wait for URL to update (include should be removed).
    $this->assertSession()->waitForElementRemoved('css', '.include-item');

    // Verify URL no longer contains include parameter.
    $this->assertSession()->pageTextNotContains('include=uid');
  }

  /**
   * Tests that the API request can be executed and shows a response.
   */
  public function testExecuteRequest(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet(Url::fromRoute('jsonapi_query_builder.app'));

    $this->assertReactAppInitializes();

    // Select node--article.
    $page = $this->getSession()->getPage();
    $page->find('css', '#entity-type-select')->selectOption('node');
    $this->assertSession()->waitForElement('css', '#bundle-select');
    $page->find('css', '#bundle-select')->selectOption('article');

    // Wait for fields to load.
    $this->assertSession()->waitForElement('css', '.fields-list');

    // Click "Execute Query" button.
    $this->getExecuteQueryButton()->click();

    // Wait for response to load.
    $this->assertSession()->waitForElement('css', '.response-display');

    // Display raw response body.
    $raw_response_locator = '//div[@data-testid="response-display"]//button[normalize-space(text())="Raw"]';
    $this->assertSession()->waitForElement('xpath', $raw_response_locator);
    $page->find('xpath', $raw_response_locator)->click();
    $raw_response_display = $this->getSession()->getPage()->find('css', '.response-display .language-json')->getText();

    // Verify response contains data.
    $this->assertStringContainsString('data', $raw_response_display);
    $this->assertStringContainsString('type', $raw_response_display);
    $this->assertStringContainsString('node--article', $raw_response_display);

    // Verify response contains article titles.
    $this->assertStringContainsString('Test Article 1', $raw_response_display);
  }

  /**
   * Waits for the React app to initialize and throws an error if it does not.
   */
  private function assertReactAppInitializes(): void {
    // Check that the main JS file has been attached.
    $this->assertSession()->responseContains('jsonapi_query_builder/js/react-app/build/static/js/');

    // Check for a key UI element to confirm the app has loaded.
    $this->assertNotNull(
      $this->assertSession()->waitForElement('css', '#jsonapi-query-builder-root .entity-type-selector'),
      'CSS locator "#jsonapi-query-builder-root .entity-type-selector" not found. React app may not have loaded properly.',
    );
  }

  /**
   * Asserts that a route with the given name exists.
   */
  private function assertRouteExists(string $name): void {
    /** @var \Drupal\Core\Routing\RouteProviderInterface $route_provider */
    $route_provider = \Drupal::service('router.route_provider');

    try {
      $route_provider->getRouteByName($name);
    }
    catch (RouteNotFoundException) {
      $this->fail(sprintf('Route "%s" not found.', $name));
    }
  }

  /**
   * Gets the "Execute Query" button element.
   */
  private function getExecuteQueryButton(): NodeElement {
    $element = $this->getSession()
      ->getPage()
      ->find('xpath', '//button[@data-testid="execute-button"]');

    $this->assertInstanceOf(NodeElement::class, $element, '"Execute Query" button not found.');

    return $element;
  }

  /**
   * Clicks on the specified panel tab and waits for it to become active.
   */
  public function clickPanelTab(string $name): void {
    $name = ucfirst($name);

    $tab = $this->getSession()
      ->getPage()
      ->find(
        'xpath',
        '//button[contains(@class, "query-panel-tab") and normalize-space(text())="' . $name . '"]',
      );

    $this->assertInstanceOf(NodeElement::class, $tab, '"' . $name . '" tab not found.');

    $tab->click();

    $h4_locator = '//*[@id="jsonapi-query-builder-root"]//h4[starts-with(normalize-space(text()), "' . $name . '")]';
    $h4_element = $this->assertSession()->waitForElement('xpath', $h4_locator);

    $this->assertInstanceOf(
      NodeElement::class,
      $h4_element,
      '"' . $name . '" panel did not become active.',
    );
  }

}
