# JSON:API Query Builder Module

This Drupal module provides a visual explorer for the JSON:API with a React-based interface for building and testing API requests.

## Development and Contribution Information

### Module Structure

- `/src/Controller/` - PHP controller for the explorer page
- `/js/react-app/src/` - React application source code
- `/js/react-app/src/components/` - React components
- `/templates/` - Twig templates
- `/tests/` - Unit and functional tests

### Building the React App

After making changes to the React code in `/js/react-app/src/`, you need to build the app for the changes to appear:

```bash
# Navigate to the React app directory
cd js/react-app

# Install dependencies (if needed)
yarn install

# Build the app
yarn build
```

Alternatively, you can use the included build script:

```bash
cd js
./build.sh
```

You can also use the Composer script:

```bash
composer build-js
```

### Clearing Drupal Cache

After building the React app, clear Drupal's cache for the changes to be visible:

```bash
# Using Drush
drush cr
```

### Running Tests

The module includes comprehensive tests that can be run using Composer scripts:

```bash
# Run all tests (coding standards, static analysis, and PHP tests)
composer test

# Run only coding standards checks
composer phpcs

# Run static analysis with PHPStan
composer phpstan

# Run only unit tests
composer phpunit

# Run only functional tests
composer phpunit-functional

# Fix coding standards automatically where possible
composer phpcbf

# Run JavaScript tests
composer test-js
```

To run all tests including JavaScript tests:

```bash
composer test-all
```

This runs coding standards checks, static analysis, PHP tests, functional JavaScript tests, and React component tests.

### UI Components Overview

Key UI components include:
- **App.js**: Main component managing state and layout
- **EntityTypeSelector.js**: Entity type and bundle selection
- **FieldsSelector.js**: Field selection with relationship field support
- **IncludeSelector.js**: Relationship inclusion management
- **FilterSelector.js**: Query filtering including relationship fields
- **SortSelector.js**: Result sorting control
- **UrlDisplay.js**: Shows the current JSON:API URL
- **ResponseDisplay.js**: Displays the API response
- **AuthSection.js**: Authentication documentation
- **BookmarksSection.js**: Saved queries management
- **RequestHistorySection.js**: Previous requests history

### Key Features

- Built with React, integrated into Drupal's admin UI
- Field and relationship discovery via JSON:API Schema
- Interactive query builder with fields, includes, filters, and sorting
- Relationship field support with dot notation (e.g., `user.name`)
- URL and cURL command generation for API requests
- Authentication documentation for API integration
- Session-based authentication (no Bearer token input required)
- Request history and bookmarks
- Responsive design with mobile support
- Accessibility features and keyboard navigation

### Drupal Integration

- Uses Drupal routing with route `jsonapi_query_builder.explorer`
- Requires 'access jsonapi query builder' permission
- Utilizes JSON:API Schema module for entity type discovery
- No configuration settings required
- Help text provided through `hook_help()`
- Asset libraries defined in `jsonapi_query_builder.libraries.yml`

### Contributing

Contributions to this module are welcome. Please follow these steps:

1. Review the CONTRIBUTING.md file for guidelines
2. Ensure code follows Drupal coding standards
3. Add tests for new functionality
4. Verify all tests pass with `composer test`
5. Submit a patch or merge request on Drupal.org

### Dependency Management

- PHP dependencies are managed in `composer.json`
- Production dependencies are listed under `require`
- Development dependencies are listed under `require-dev`
- JavaScript dependencies are managed in `js/react-app/package.json`
- Production React dependencies are under `dependencies`
- Development React dependencies are under `devDependencies`

### Technical Implementation Notes

Related entities implementation:
- When relationships are included, the app fetches schema data for related entities
- Fields are made available with dot notation (relationship.field) for selection
- Relationship fields are visually distinct with styling
- When a relationship is removed, associated fields, filters, and sort criteria are automatically cleaned up
- Field tooltips provide metadata about fields, including from related entities
- Notifications appear when new relationships are added

Filtering implementation:
- Supports standard Drupal JSON:API filtering patterns
- Includes operators for equality, path-based filtering, and string matching
- String operators: CONTAINS, STARTS_WITH, ENDS_WITH
- Support for comparison operators (<, >, etc.)
- Automatically formats filter URLs according to JSON:API specification

Authentication implementation:
- Uses Drupal's session cookies for authentication (no Bearer token needed)
- Authentication tab provides documentation for external API access
- Includes code examples in JavaScript and cURL formats
- Examples show how to obtain and use Bearer tokens for API access outside Drupal

### Testing

- PHP unit tests validate controller functionality
- Functional tests ensure proper route access and permission handling
- JavaScript tests for React components use Jest and React Testing Library
- Screenshot tests validate UI layout and responsive design
- End-to-end tests that validate the full application within a real Drupal environment
- Run `composer test-all` to execute the standard test suite
- Run `composer test-e2e-install` to set up a test Drupal site for end-to-end testing
- Run `composer test-e2e-interactive` to run end-to-end tests with a visible browser
- Run `composer test-e2e` to run end-to-end tests in headless mode (for CI/CD)
- Run `composer test-e2e-full` to automatically setup Drupal, install the module, and run tests in one command
- The automated tests verify that the module can be installed and the app loads correctly
