import React from 'react';
import { Prism as SyntaxHighlighter } from 'react-syntax-highlighter';
import { materialLight } from 'react-syntax-highlighter/dist/esm/styles/prism';

const ExamplesTab = ({ baseUrl, expandedExample, setExpandedExample, copyToClipboard }) => {
  // Helper function to render a code example accordion
  const renderCodeAccordion = (title, key, code, language = 'bash') => {
    return (
      <div style={{
        marginBottom: '12px',
        border: '1px solid var(--explorer-border)',
        borderRadius: '4px',
        overflow: 'hidden'
      }}>
        <div
          style={{
            padding: '12px 16px',
            backgroundColor: 'var(--explorer-background-secondary)',
            borderBottom: expandedExample === key ? '1px solid var(--explorer-border)' : 'none',
            cursor: 'pointer',
            display: 'flex',
            justifyContent: 'space-between',
            alignItems: 'center',
            fontWeight: '500'
          }}
          onClick={() => setExpandedExample(expandedExample === key ? '' : key)}
        >
          <span>{title}</span>
          {expandedExample === key ? (
            <button
              className="button button--small"
              onClick={(e) => {
                e.stopPropagation();
                copyToClipboard(code);
              }}
              title="Copy example code"
            >
              Copy
            </button>
          ) : (
            <span style={{ fontSize: '0.9em', opacity: 0.6 }}>Click to expand</span>
          )}
        </div>

        {expandedExample === key && (
          <div style={{
            display: 'block',
            padding: '0',
            backgroundColor: 'var(--explorer-background-tertiary)'
          }}>
            <SyntaxHighlighter
              language={language}
              style={materialLight}
              wrapLines={true}
              customStyle={{
                fontSize: '0.8rem',
                margin: 0,
                padding: '16px',
                backgroundColor: 'var(--explorer-background-tertiary)'
              }}
            >
              {code}
            </SyntaxHighlighter>
          </div>
        )}
      </div>
    );
  };

  return (
    <div className="layout-row" data-testid="code-examples-tab">
      <div className="layout-column layout-column--full">
        <div className="panel">
          <div className="gin-layer-wrapper">
            <h3 className="heading">Advanced JSON:API examples</h3>

            <div style={{ display: 'flex', flexDirection: 'column', gap: '12px' }}>
              {/* Render all examples using our helper function */}
              {renderCodeAccordion('Basic filtering', 'filtering',
                `# Basic filter examples
curl -X GET "${baseUrl}node/article?filter[status]=1&filter[title][CONTAINS]=Example" \\
  -H "Accept: application/vnd.api+json"

# Filter with operators
curl -X GET "${baseUrl}node/article?filter[created][>]=2023-01-01" \\
  -H "Accept: application/vnd.api+json"`
              )}

              {renderCodeAccordion('Advanced filtering with groups', 'advanced_filtering',
                `# Filter with AND/OR groups (complex conditions)
# Find articles that are either (published AND contain "news") OR (have a specific tag)

curl -X GET "${baseUrl}node/article?\\
filter[group1][conjunction]=AND\\
&filter[condition1][condition][memberOf]=group1\\
&filter[condition1][condition][field]=status\\
&filter[condition1][condition][value]=1\\
&filter[condition2][condition][memberOf]=group1\\
&filter[condition2][condition][field]=title\\
&filter[condition2][condition][operator]=CONTAINS\\
&filter[condition2][condition][value]=news\\
&filter[group2][conjunction]=OR\\
&filter[group2][memberOf]=\\
&filter[condition3][condition][memberOf]=group2\\
&filter[condition3][condition][field]=field_tags.name\\
&filter[condition3][condition][value]=featured" \\
  -H "Accept: application/vnd.api+json"`
              )}

              {renderCodeAccordion('Including related resources', 'includes',
                `# Include related entities
curl -X GET "${baseUrl}node/article?include=uid,field_tags" \\
  -H "Accept: application/vnd.api+json"`
              )}

              {renderCodeAccordion('Sparse fieldsets', 'fields',
                `# Request only specific fields
curl -X GET "${baseUrl}node/article?fields[node--article]=title,created,status&fields[user--user]=name" \\
  -H "Accept: application/vnd.api+json"`
              )}

              {renderCodeAccordion('Sorting results', 'sorting',
                `# Sort by multiple fields
curl -X GET "${baseUrl}node/article?sort=-created,title" \\
  -H "Accept: application/vnd.api+json"`
              )}

              {renderCodeAccordion('Pagination', 'pagination',
                `# Paginate results
curl -X GET "${baseUrl}node/article?page[offset]=10&page[limit]=5" \\
  -H "Accept: application/vnd.api+json"`
              )}

              {renderCodeAccordion('Creating relationships', 'relationships',
                `# Create content with relationships
curl -X POST "${baseUrl}node/article" \\
  -H "Authorization: Bearer YOUR_ACCESS_TOKEN" \\
  -H "Content-Type: application/vnd.api+json" \\
  -H "Accept: application/vnd.api+json" \\
  -d '{
    "data": {
      "type": "node--article",
      "attributes": {
        "title": "Article with Tags",
        "body": {
          "value": "This article has tags attached.",
          "format": "full_html"
        }
      },
      "relationships": {
        "field_tags": {
          "data": [
            {
              "type": "taxonomy_term--tags",
              "id": "TAG_UUID_HERE"
            }
          ]
        }
      }
    }
  }'`
              )}

              {renderCodeAccordion('File upload example', 'file-upload',
                `# Create a file entity (2-step process)
# Step 1: Upload the file
curl -X POST "${baseUrl}file/file" \\
  -H "Authorization: Bearer YOUR_ACCESS_TOKEN" \\
  -H "Content-Type: application/octet-stream" \\
  -H "Content-Disposition: file; filename=\\"example.jpg\\"" \\
  -H "Accept: application/vnd.api+json" \\
  --data-binary @/path/to/example.jpg

# Step 2: Create a media entity referencing the file
curl -X POST "${baseUrl}media/image" \\
  -H "Authorization: Bearer YOUR_ACCESS_TOKEN" \\
  -H "Content-Type: application/vnd.api+json" \\
  -H "Accept: application/vnd.api+json" \\
  -d '{
    "data": {
      "type": "media--image",
      "attributes": {
        "name": "My Image"
      },
      "relationships": {
        "field_media_image": {
          "data": {
            "type": "file--file",
            "id": "FILE_UUID_FROM_STEP_1"
          }
        }
      }
    }
  }'`
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ExamplesTab;
