#!/usr/bin/env bash

#
# release.sh -- Automates the release process for this module.
#
# This script creates a temporary release branch, runs the build for front-end
# assets, commits build artifacts, creates a release tag, returns to the main
# development branch, and provides instructions for remaining manual actions to
# make the release live. It can be run from anywhere.
#
# Usage:
#   ./scripts/release.sh [-h | --help] [-H | --head <branch>] [--dry-run] [-f | --force] <tag_name>
#
# Options:
#   -h, --help            Show this help message and exit.
#   -H, --head <branch>   (Optional) The branch to cut the release from. Defaults to the current branch.
#   --dry-run             Print the operations that would be performed, but do not execute them.
#   -f, --force           Ignore precondition checks (for testing/debugging only).
#   <tag_name>            The name of the release tag to create.
#
# Examples:
#   ./scripts/release.sh 2.1.0
#   ./scripts/release.sh --head 2.1.x 2.1.0
#
# This script is intended for maintainers and release managers.
#

set -e

BUILD_DIR="js/react-app/build"

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_ROOT="$(git -C "$SCRIPT_DIR" rev-parse --show-toplevel)"
cd "$REPO_ROOT"

usage() {
  echo "Usage: $0 [-h | --help] [-H | --head <branch>] [--dry-run] [-f | --force] <tag_name>"
  echo "  -h, --help            Show this help message and exit."
  echo "  -H, --head <branch>   (Optional) The branch to return to after release. Defaults to current branch."
  echo "  --dry-run             Print the operations that would be performed, but do not execute them."
  echo "  -f, --force           Ignore precondition checks (for testing/debugging only)."
  echo "  <tag_name>            The name of the release tag to create."
}

usage_error() {
  echo "Error: $*" >&2;
  usage
  exit 1
}

precondition_error() {
  if [ "$FORCE" -eq 1 ]; then
	echo "- Ignoring error: $*";
    return
  fi

  echo "Error: $*" >&2;
  exit 1
}

fatal_error() {
  echo "Error: $*" >&2;
  exit 1
}

DRY_RUN=0
FORCE=0

parse_args() {
  HEAD_BRANCH=""

  while [[ $# -gt 0 ]]; do
    case "$1" in
      --dry-run)
        DRY_RUN=1
        ;;
      --head|-H)
        shift
        HEAD_BRANCH="$1"
        ;;
      --force|-f)
        FORCE=1
        ;;
      --help|-h)
        usage
        exit 0
        ;;
      --)
        shift
        break
        ;;
      -* )
        usage_error "Unknown option: $1"
        ;;
      *)
        break
        ;;
    esac
    shift
  done

  # Check for missing or empty <tag_name> argument:
  if [ $# -ne 1 ] || [ -z "$1" ]; then
    usage_error "Missing required <tag_name> argument."
  fi

  TAG_NAME="$1"
}

# Prints a message and runs a command unless in dry-run mode.
# Usage: print_and_conditionally_run "Message" command [args...]
print_and_conditionally_run() {
  local msg="$1"
  shift
  echo "- $msg: $*"
  if [ "$DRY_RUN" -eq 1 ]; then
    return
  fi
  "$@"
}

set_default_head_branch() {
  if [ -z "$HEAD_BRANCH" ]; then
    HEAD_BRANCH="$(git branch --no-color | grep '^\* ' | grep -v 'no branch' | sed 's/^* //g')"
  fi
}

assert_preconditions() {
  # Ensure the tag does not already exist.
  if git rev-parse "refs/tags/$TAG_NAME" >/dev/null 2>&1; then
    precondition_error "Tag '$TAG_NAME' already exists. Delete it or choose a different tag name."
  fi

  # Ensure the release branch does not already exist.
  if git rev-parse "refs/heads/release/$TAG_NAME" >/dev/null 2>&1; then
    precondition_error "Release branch 'release/$TAG_NAME' already exists. Delete it or choose a different tag name."
  fi

  # Ensure the HEAD branch exists.
  if ! git show-ref --verify --quiet "refs/heads/$HEAD_BRANCH"; then
    precondition_error "HEAD branch '$HEAD_BRANCH' does not exist. Create it or specify a valid branch with --head."
  fi

  # Ensure working tree is clean.
  if [ -n "$(git status --porcelain)" ]; then
    precondition_error "Working tree is not clean. Commit or stash your changes before running the release script."
  fi
}

create_and_checkout_release_branch() {
  RELEASE_BRANCH="release/$TAG_NAME"
  print_and_conditionally_run "Create and checkout release branch" \
    git checkout -b "$RELEASE_BRANCH"
}

run_build() {
  print_and_conditionally_run "Run build script" \
    "$REPO_ROOT/js/build.sh"
  echo
}

add_and_commit_build() {
  print_and_conditionally_run "Force add build artifacts" \
    git add --force "$BUILD_DIR"
  print_and_conditionally_run "Commit build artifacts" \
    git commit -m "$TAG_NAME release"
}

create_tag() {
  print_and_conditionally_run "Create tag" \
    git tag -a "$TAG_NAME" -m "$TAG_NAME"
}

checkout_head_branch() {
  print_and_conditionally_run "Checkout head branch" \
    git checkout "$HEAD_BRANCH"
}

delete_release_branch() {
  print_and_conditionally_run "Delete release branch" \
    git branch -D "$RELEASE_BRANCH"
}

restore_build_directory() {
  print_and_conditionally_run "Restore build directory" \
    git checkout "$TAG_NAME" -- "$BUILD_DIR" && git reset HEAD -- "$BUILD_DIR"
}

validate_artifact() {
  # Make sure the build directory exists in the tag.
  if ! git ls-tree -d --name-only "$TAG_NAME" "$BUILD_DIR" >/dev/null 2>&1; then
    echo
    echo "Build output directory '$BUILD_DIR' is missing from tag '$TAG_NAME'."
    echo
    exit 1
  fi
}

print_summary() {
  echo
  if [ "$DRY_RUN" -eq 1 ]; then
    echo "Dry run complete. No changes were made."
  else
    echo "Release process complete."
    echo "- Created and checked out branch: $RELEASE_BRANCH"
    echo "- Ran build and committed build artifacts"
    echo "- Created tag: $TAG_NAME"
    echo "- Returned to HEAD branch: $HEAD_BRANCH"
    echo "- Deleted release branch: $RELEASE_BRANCH"
    echo "- Restored build directory"
    echo "- Validated build artifact"
  fi
  echo
  echo "Manual next steps:"
  echo "- Test the tag locally: git checkout $TAG_NAME"
  echo "- Push the tag to remote: git push origin $TAG_NAME"
  echo "- Create a Drupal.org release node for tag $TAG_NAME"
  echo
  echo "Note: Once you checkout the release tag, when you return to a"
  echo "branch, Git will delete the build directory. To restore it, run:"
  echo
  echo "  git checkout $TAG_NAME -- $BUILD_DIR && git reset HEAD -- $BUILD_DIR"
  echo
}

main() {
  parse_args "$@"
  set_default_head_branch
  assert_preconditions
  create_and_checkout_release_branch
  run_build
  add_and_commit_build
  create_tag
  checkout_head_branch
  delete_release_branch
  restore_build_directory
  validate_artifact
  print_summary
}

main "$@"
