<?php

namespace Drupal\Tests\jsonapi_role_access\Functional;

use Drupal\Core\Url;
use Drupal\Tests\ApiRequestTrait;
use Drupal\Tests\BrowserTestBase;
use Symfony\Component\HttpFoundation\Response;

use function PHPUnit\Framework\assertEquals;

/**
 * Tests the core role based access functionality of the module.
 *
 * @group jsonapi_role_access
 */
class JsonApiRoleAccessRestrictionTest extends BrowserTestBase {
  use ApiRequestTrait;

  /**
   * The name of the tested JSON:API resource type.
   *
   * @var string
   */
  protected static $resourceTypeName = 'node--article';

  /**
   * The name of a role that is assigned to the user.
   *
   * @var string
   */
  protected static $userAssignedRole = 'user_assigned_role';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'path',
    'jsonapi',
    'jsonapi_role_access',
  ];

  /**
   * A user with authenticated permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $user;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->createRole(['access content'], static::$userAssignedRole);
    $this->user = $this->drupalCreateUser(['access content']);
    $this->user->addRole(static::$userAssignedRole);
    $this->user->save();
    $this->drupalLogin($this->user);
  }

  /**
   * Test the role based access restriction to JSON:API resources.
   */
  public function testRoleBasedAccessRestriction() {
    // Create a basic article for access via JSON:API.
    $this->createContentType(['type' => 'article']);
    // Rebuild router so JSON:API routes are available for the new content type.
    \Drupal::service('router.builder')->rebuildIfNeeded();
    $node = $this->createNode([
      'type' => 'article',
      'title' => 'test123',
      'status' => 1,
    ]);
    $url = Url::fromRoute(sprintf('jsonapi.%s.individual', static::$resourceTypeName), ['entity' => $node->uuid()]);
    $requestOptions = [
      'headers' => [
        'Authorization' => 'Basic ' . base64_encode($this->user->name->value . ':' . $this->user->passRaw),
        'Accept' => 'application/vnd.api+json',
      ],
    ];
    // By default authenticated users are allowed to access the API:
    $response = $this->makeApiRequest('GET', $url, $requestOptions);
    assertEquals(Response::HTTP_OK, $response->getStatusCode());

    // Remove authenticated user access but allow the user assigned role:
    \Drupal::configFactory()->getEditable('jsonapi_role_access.settings')
      ->set('roles', [static::$userAssignedRole => static::$userAssignedRole])
      ->save();
    // Since the user also has the test role, the request should be successful:
    $response = $this->makeApiRequest('GET', $url, $requestOptions);
    assertEquals(Response::HTTP_OK, $response->getStatusCode());

    // Remove user assigned role access:
    \Drupal::configFactory()->getEditable('jsonapi_role_access.settings')
      ->set('roles', [])
      ->save();
    // The user doesn't have the right roles anymore, so the request should
    // return a 403 error:
    $response = $this->makeApiRequest('GET', $url, $requestOptions);
    assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());

    // Restrict the user assigned role:
    \Drupal::configFactory()->getEditable('jsonapi_role_access.settings')
      ->set('roles', [static::$userAssignedRole => static::$userAssignedRole])
      ->set('negate', TRUE)
      ->save();
    // Since the user assigned role is restricted, the request should return a
    // 403 error:
    $response = $this->makeApiRequest('GET', $url, $requestOptions);
    assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());
  }

}
