<?php

namespace Drupal\jsonapi_role_access\EventSubscriber;

use Symfony\Component\HttpKernel\Event\RequestEvent;
use Drupal\Core\Session\AccountProxy;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Checks subscriptions for user role access for json api API keys.
 */
class CheckUserRolePermissionEvent implements EventSubscriberInterface {

  /**
   * The config object.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * Constructs this factory object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxy $currentUser
   *   The account object.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    protected AccountProxy $currentUser,
  ) {
    $this->config = $configFactory->get('jsonapi_role_access.settings');
  }

  /**
   * Returns an array of event names this subscriber wants to listen to.
   *
   * The array keys are event names and the value can be:
   *
   *  * The method name to call (priority defaults to 0)
   *  * An array composed of the method name to call and the priority
   *  * An array of arrays composed of the method names to call and respective
   *    priorities, or 0 if unset
   *
   * For instance:
   *
   *  * ['eventName' => 'methodName']
   *  * ['eventName' => ['methodName', $priority]]
   *  * ['eventName' => [['methodName1', $priority], ['methodName2']]]
   *
   * @return array
   *   The event names to listen to.
   */
  public static function getSubscribedEvents() {
    $events[KernelEvents::REQUEST][] = ['checkUserRoleAccess', 30];
    return $events;
  }

  /**
   * Check if user has subscription, if not redirect to subscription page.
   */
  public function checkUserRoleAccess(RequestEvent $event) {
    // If the request is an XMLHttpRequest or the route is not a jsonapi route,
    // or the route is one of the jsonapi settings routes, return early:
    if (
      $event->getRequest()->isXmlHttpRequest() ||
      !$this->strStartsWith($event->getRequest()->get('_route'), 'jsonapi') ||
      in_array($event->getRequest()->get('_route'), [
        'jsonapi.settings',
        'jsonapi_extras.settings',
        'jsonapi_role_access.config',
      ]
      )) {
      return;
    }

    $negate = $this->config->get('negate');
    $roles = $this->config->get('roles');
    $userRoles = $this->currentUser->getRoles();
    // Check if the user has any of the selected roles:
    $hasIntersection = !empty(array_intersect($roles, $userRoles));
    // Based on the conditions, throw an access denied exception:
    if (!$negate && !$hasIntersection || $negate && $hasIntersection) {
      throw new AccessDeniedHttpException();
    }
  }

  /**
   * Polyfill for PHP 7.0 and later.
   *
   * @param string $haystack
   *   The source string.
   * @param string $needle
   *   The needle string.
   *
   * @return bool
   *   Return true if found else false.
   */
  private function strStartsWith($haystack, $needle) {
    if (function_exists('str_starts_with')) {
      return str_starts_with($haystack, $needle);
    }
    return \strncmp($haystack, $needle, \strlen($needle)) === 0;
  }

}
