<?php

namespace Drupal\jsonrpc\Attribute;

use Drupal\Component\Plugin\Attribute\Plugin;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\jsonrpc\JsonRpcMethodDefinition;
use Drupal\jsonrpc\MethodInterface;

/**
 * The JsonRpcMethod attribute.
 */
#[\Attribute(\Attribute::TARGET_CLASS)]
class JsonRpcMethod extends Plugin {

  /**
   * The class method to call.
   *
   * Optional. If the method ID is 'foo.bar', this defaults to 'bar'. If the
   * method ID does not contain a dot (.), defaults to 'execute'.
   *
   * @var string|null
   */
  public ?string $call = NULL;

  /**
   * Constructs a JsonRpcMethod attribute.
   *
   * @param string $id
   *   The method ID.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $usage
   *   A description of how to use the method.
   * @param array|null $output
   *   (Optional.) A copy of the output schema from the JSON-RPC method.
   * @param array<string> $access
   *   (Optional.) An array of permissions to check for access.
   * @param array|null $params
   *   (Optional.) An array of JsonRpcParameterDefinitions.
   *   These are the parameters used when calling the endpoint.
   *   The array is keyed by parameter name.
   * @param array $responseHeaders
   *   (Optional.) Key value pairs for setting headers in the response.
   *   Batch requests will only contain the headers appearing on **all** of the
   *   responses involved.
   */
  public function __construct(
    public readonly string $id,
    public readonly TranslatableMarkup $usage,
    public array|null $output = NULL,
    public readonly array $access = [],
    public array|null $params = NULL,
    public readonly array $responseHeaders = [],
  ) {
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function get(): MethodInterface {
    return new JsonRpcMethodDefinition(parent::get() + [
      'call' => $this->call,
      'usage' => $this->usage,
      'output' => $this->output,
      'access' => $this->access,
      'params' => $this->params,
      'responseHeaders' => $this->responseHeaders,
    ]);
  }

}
