<?php

namespace Drupal\jsonrpc\Attribute;

use Drupal\Component\Plugin\Attribute\Plugin;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\jsonrpc\ParameterDefinitionInterface;
use Drupal\jsonrpc\ParameterFactory\RawParameterFactory;
use Drupal\jsonrpc\ParameterFactoryInterface;

/**
 * The JsonRpcParameterDefinition attribute.
 */
#[\Attribute(\Attribute::TARGET_CLASS)]
class JsonRpcParameterDefinition extends Plugin implements ParameterDefinitionInterface {

  /**
   * Constructs a JsonRpcMethod attribute.
   *
   * @param string $id
   *   The name of the parameter.
   * @param array|null $schema
   *   (Either schema or factory is required.) The parameter schema.
   *   This will be converted with json_encode().
   * @param string|null $factory
   *   (Either schema or factory is required.) The parameter factory class.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|null $description
   *   (Optional) A description of the parameter.
   * @param bool|null $required
   *   (Optional)  Whether the parameter is required. Defaults to FALSE.
   */
  public function __construct(
    public readonly string $id,
    public readonly ?array $schema = NULL,
    public readonly ?string $factory = NULL,
    public readonly ?TranslatableMarkup $description = NULL,
    public readonly ?bool $required = FALSE,
  ) {
    if (is_null($schema) && is_null($factory)) {
      throw new InvalidPluginDefinitionException($id, "Every JsonRpcParameterDefinition must define either a factory or a schema.");
    }
    if ($factory && !is_subclass_of($factory, ParameterFactoryInterface::class)) {
      throw new InvalidPluginDefinitionException($id, "Parameter factories must implement ParameterFactoryInterface.");
    }
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getFactory(): string {
    return $this->factory ?? RawParameterFactory::class;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getDescription(): ?TranslatableMarkup {
    return $this->description;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function isRequired(): bool {
    return $this->required ?? FALSE;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getSchema(): array {
    return $this->schema ?? [];
  }

  /**
   * Returns the instance.
   */
  #[\Override]
  public function get(): JsonRpcParameterDefinition {
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getId(): string {
    return $this->id;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function setClass($class): void {
    $this->class = $class;
  }

}
