<?php

declare(strict_types=1);

namespace Drupal\Tests\jsonrpc\Functional;

use Drupal\Component\Serialization\Json;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;

/**
 * Base test for response headers functionality.
 *
 * @group jsonrpc
 */
abstract class ResponseHeadersTestBase extends JsonRpcTestBase {

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function setUp(): void {
    parent::setUp();
    $this->grantPermissions(Role::load(RoleInterface::ANONYMOUS_ID), [
      'use jsonrpc services',
      'access content',
    ]);
  }

  /**
   * Tests response headers.
   */
  public function testResponseHeaders(): void {
    $batch_request = [
      [
        'jsonrpc' => '2.0',
        'method' => $this->getSecondMethodId(),
        'id' => 'second',
      ],
      [
        'jsonrpc' => '2.0',
        'method' => $this->getFirstMethodId(),
        'id' => 'first',
      ],
    ];

    $response = $this->postRpc($batch_request);
    $this->assertSame('oof', $response->getHeader('foo')[0]);
    $this->assertSame('', $response->getHeader('hello')[0]);
    $this->assertEmpty($response->getHeader('lorem'));
    $this->assertEmpty($response->getHeader('bye'));
  }

  /**
   * Ensures that a response not matching the schema produces invalid results.
   */
  public function testInvalidSchema(): void {
    $request = [
      'jsonrpc' => '2.0',
      'method' => $this->getThirdMethodId(),
      'id' => 'third',
    ];
    $response = $this->postRpc($request);
    $contents = $response->getBody();
    $this->assertSame(-32603, Json::decode($contents)['error']['code']);
  }

  /**
   * Get the first method ID.
   *
   * @return string
   *   The method ID.
   */
  abstract protected function getFirstMethodId(): string;

  /**
   * Get the second method ID.
   *
   * @return string
   *   The method ID.
   */
  abstract protected function getSecondMethodId(): string;

  /**
   * Get the third method ID.
   *
   * @return string
   *   The method ID.
   */
  abstract protected function getThirdMethodId(): string;

}
