<?php

declare(strict_types=1);

namespace Drupal\jsonrpc_discovery\Normalizer;

use Drupal\Component\Annotation\AnnotationInterface;
use Drupal\Component\Assertion\Inspector;
use Drupal\Core\Url;
use Drupal\jsonrpc\Annotation\JsonRpcMethod;
use Drupal\jsonrpc\Annotation\JsonRpcParameterDefinition;
use Drupal\jsonrpc\Attribute\JsonRpcParameterDefinition as AttributeJsonRpcParameterDefinition;
use Drupal\jsonrpc\JsonRpcMethodDefinition;
use Drupal\jsonrpc\MethodInterface;
use Drupal\jsonrpc\ParameterDefinitionInterface;
use Drupal\serialization\Normalizer\NormalizerBase;

/**
 * The normalizer class for definition objects (annotations and attributes).
 */
class DefinitionNormalizer extends NormalizerBase {

  const DEPTH_KEY = self::class . '_depth';

  /**
   * {@inheritdoc}
   */
  protected $format = 'json';

  /**
   * The supported interfaces and classes.
   *
   * @var array
   *
   * @todo Remove when support for D10.0 is dropped.
   * https://www.drupal.org/node/3359695
   */
  protected array $supportedInterfaceOrClass = [
    JsonRpcMethod::class,
    JsonRpcParameterDefinition::class,
    AttributeJsonRpcParameterDefinition::class,
    JsonRpcMethodDefinition::class,
  ];

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getSupportedTypes(?string $format): array {
    return [
      JsonRpcMethod::class => TRUE,
      JsonRpcParameterDefinition::class => TRUE,
      AttributeJsonRpcParameterDefinition::class => TRUE,
      JsonRpcMethodDefinition::class => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function normalize(mixed $object, mixed $format = NULL, array $context = []): array|\ArrayObject|bool|float|int|string|null {
    $attributes = [];
    foreach ($object as $key => $value) {
      switch ($key) {
        case 'id':
        case 'call':
        case 'access':
          break;

        default:
          $child = $value instanceof AnnotationInterface ? $value->get() : $value;
          if (isset($context[static::DEPTH_KEY]) && ($child instanceof AnnotationInterface || (is_array($child)) && Inspector::assertAllObjects($child, AnnotationInterface::class))) {
            if ($context[static::DEPTH_KEY] === 0) {
              break;
            }
            $context[static::DEPTH_KEY] -= 1;
          }
          $attributes[$key] = $this->serializer->normalize($child, $format, $context);
      }
    }
    $normalized = [
      'type' => static::getDefinitionType($object),
      // Annotations have a getId() method.
      // Attributes have an id() method.
      'id' => method_exists($object, 'getId') ? $object->getId() : $object->id(),
      'attributes' => array_filter($attributes),
    ];
    if ($object instanceof MethodInterface) {
      $self = Url::fromRoute('jsonrpc.method_resource', [
        'method_id' => $object->id(),
      ])->setAbsolute()->toString(TRUE);
      $collection = Url::fromRoute('jsonrpc.method_collection')->setAbsolute()->toString(TRUE);
      $this->addCacheableDependency($context, $self);
      $this->addCacheableDependency($context, $collection);
      $normalized['links'] = [
        'self' => $self->getGeneratedUrl(),
        'collection' => $collection->getGeneratedUrl(),
      ];
    }
    if ($object instanceof ParameterDefinitionInterface) {
      $normalized['schema'] = $object->getSchema();
    }
    return $normalized;
  }

  /**
   * Extract the definition type.
   *
   * @param mixed $definition
   *   The definition object.
   *
   * @return string
   *   The type.
   */
  protected static function getDefinitionType(mixed $definition): string {
    switch ($definition::class) {
      case JsonRpcMethod::class:
      case JsonRpcMethodDefinition::class:
        return 'JsonRpcMethod';

      case JsonRpcParameterDefinition::class:
      case AttributeJsonRpcParameterDefinition::class:
        return 'JsonRpcParameterDefinition';

      default:
        return $definition::class;
    }
  }

}
