<?php

declare(strict_types=1);

namespace Drupal\jsonrpc;

use Drupal\Component\Plugin\Definition\PluginDefinition;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Defines a JSON-RPC method plugin definition.
 */
class JsonRpcMethodDefinition extends PluginDefinition implements MethodInterface {

  /**
   * The class method to call.
   *
   * @var string|null
   */
  public ?string $call = NULL;

  /**
   * The method usage description.
   *
   * @var \Drupal\Core\StringTranslation\TranslatableMarkup
   */
  public TranslatableMarkup $usage;

  /**
   * The method output schema.
   *
   * @var array|null
   */
  public array|null $output = NULL;

  /**
   * The access permissions array.
   *
   * @var array
   */
  public array $access = [];

  /**
   * The method parameters.
   *
   * @var array|null
   */
  public array|null $params = NULL;

  /**
   * The response headers.
   *
   * @var array
   */
  public array $responseHeaders = [];

  /**
   * Constructs a new JsonRpcMethodDefinition.
   *
   * @param array $definition
   *   The plugin definition array.
   */
  public function __construct(array $definition = []) {
    // Set properties from the definition array.
    foreach ($definition as $key => $value) {
      if (property_exists($this, $key)) {
        $this->{$key} = $value;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function call(): string {
    if (!isset($this->call)) {
      $this->call = 'execute';
    }
    return $this->call;
  }

  /**
   * {@inheritdoc}
   */
  public function getUsage(): TranslatableMarkup {
    return $this->usage;
  }

  /**
   * {@inheritdoc}
   */
  public function getParams(): ?array {
    return $this->params;
  }

  /**
   * {@inheritdoc}
   */
  public function areParamsPositional(): bool {
    $params = $this->getParams();
    if ($params === NULL) {
      return TRUE;
    }
    return array_reduce(array_keys($params), fn($positional, $key): bool => $positional ? !is_string($key) : $positional, TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function access($operation = 'execute', ?AccountInterface $account = NULL, $return_as_object = FALSE): AccessResultInterface|bool {
    // @phpstan-ignore-next-line \Drupal calls should be avoided in classes, use dependency injection instead
    $account = $account ?: \Drupal::currentUser();
    switch ($operation) {
      case 'execute':
        if (is_callable($this->access)) {
          return call_user_func_array($this->access, [
            $operation,
            $account,
            $return_as_object,
          ]);
        }
        $access_result = AccessResult::allowed();
        foreach ($this->access as $permission) {
          $access_result = $access_result->andIf(AccessResult::allowedIfHasPermission($account, $permission));
        }
        break;

      case 'view':
        $access_result = $this->access('execute', $account, $return_as_object);
        break;

      default:
        $access_result = AccessResult::neutral();
        break;
    }
    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

}
