<?php

declare(strict_types=1);

namespace Drupal\jsonrpc\Plugin;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\jsonrpc\Annotation\JsonRpcMethod as JsonRpcMethodAnnotation;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\jsonrpc\Attribute\JsonRpcMethod;
use Drupal\jsonrpc\MethodInterface;
use Drupal\jsonrpc\ParameterFactoryInterface;

/**
 * Provides the JsonRpcMethod plugin manager.
 *
 * @internal
 */
class JsonRpcMethodManager extends DefaultPluginManager {

  /**
   * Constructs a JsonRpcMethodManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/jsonrpc/Method',
      $namespaces,
      $module_handler,
      NULL,
      JsonRpcMethod::class,
      JsonRpcMethodAnnotation::class,
    );
    $this->setCacheBackend($cache_backend, 'jsonrpc_plugins');
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function alterDefinitions(&$definitions): void {
    /** @var \Drupal\jsonrpc\Annotation\JsonRpcMethod|\Drupal\jsonrpc\Attribute\JsonRpcMethod $method */
    foreach ($definitions as &$method) {
      if (isset($method->params)) {
        $this->assertValidJsonRpcMethodPlugin($method);

        if ($method instanceof JsonRpcMethodAnnotation) {
          foreach ($method->params as $key => &$param) {
            $param->setId($key);
          }
        }
      }
      $class = $method->getClass();
      $output_schema = $class::outputSchema();
      $method->output = $output_schema;
    }
    parent::alterDefinitions($definitions);
  }

  /**
   * Asserts that the plugin class is valid.
   *
   * @param \Drupal\jsonrpc\MethodInterface $method
   *   The JSON-RPC method definition.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   */
  protected function assertValidJsonRpcMethodPlugin(MethodInterface $method): void {
    foreach ($method->params as $param) {
      if (!$param->factory && !$param->schema) {
        throw new InvalidPluginDefinitionException($method->id(), "Every JsonRpcParameterDefinition must define either a factory or a schema.");
      }
      if ($param->factory && !is_subclass_of($param->factory, ParameterFactoryInterface::class)) {
        throw new InvalidPluginDefinitionException($method->id(), "Parameter factories must implement ParameterFactoryInterface.");
      }
    }
  }

}
