<?php

namespace Drupal\keepeek\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\EntityReferenceFieldItemList;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;
use Drupal\image\ImageStyleInterface;

/**
 * Provides a base implementation for a Keepeek formatter.
 */
trait KeepeekImageFormatterTrait {

  /**
   * The asset metadata.
   *
   * @var object
   */
  protected $json;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'formatter_class' => static::class,
      'dynameek' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);
    $config = \Drupal::config('keepeek.settings');

    $form['dynameek'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Serve images from Dynameek'),
      '#description' => $this->t('If enabled, images will be served from Dynameek.'),
      '#default_value' => $this->getSetting('dynameek'),
      '#disabled' => empty($config->get('dynameek.key')),
      '#weight' => 11,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    if ($this->getSetting('dynameek')) {
      $summary[] = $this->t('Images will be served from Dynameek.');
    }

    return $summary;
  }

  /**
   * Returns the referenced entities for display.
   *
   * This method overrides the parent parameter type declaration, so we can
   * adapt the string field data type to temporary file references.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The item list.
   * @param string $langcode
   *   The language code of the referenced entities to display.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   The array of referenced entities to display, keyed by delta.
   */
  protected function getEntitiesToView(FieldItemListInterface $items, $langcode) {
    // Mimic the field item list, so it can be processed by parent class.
    $file_items = EntityReferenceFieldItemList::createInstance($items->getDataDefinition(), $items->getName(), $items->getParent());
    $file_system = \Drupal::service('file_system');
    $image_factory = \Drupal::service('image.factory');
    $use_original = empty($this->getSetting('image_style')) && empty($this->getSetting('responsive_image_style'));

    // Temporarily download every file to simulate file entity.
    foreach ($items as $delta => $item) {
      $quality_url = _keepeek_quality_url($item->value, $this->json);
      if ($use_original) {
        $uri = $quality_url;
      }
      else {
        $path = ltrim(parse_url($quality_url, PHP_URL_PATH), '/.');
        $uri = 'public://keepeek/' . $path;
        if (!file_exists($uri)) {
          $directory = dirname($uri);
          if ($file_system->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS)) {
            copy($quality_url, $uri);
          }
        }
      }

      // Only images are supported.
      $image = $image_factory->get($uri);
      if ($image->isValid()) {

        // Fill simulated file entity data, so image styles are created.
        $item->entity = File::create(['uri' => $uri]);
        $item->width = $image->getWidth();
        $item->height = $image->getHeight();
        $item->_loaded = TRUE;
        $item->_quality_url = $quality_url;
        $file_items->set($delta, $item);
      }
    }

    return parent::getEntitiesToView($file_items, $langcode);
  }

  /**
   * {@inheritdoc}
   */
  public function prepareView(array $entities_items) {
    // Skip the parent logic to process everything inside getEntitiesToView().
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    // Extract asset metadata for later usage.
    $entity = $items->getEntity();
    $field_name = $entity->bundle->entity->getFieldMap()['json'] ?? '';
    $this->json = json_decode($entity->get($field_name)->value ?? '');

    // Switch the logic between default file approach and Dynameek.
    if ($this->getSetting('dynameek')) {
      $elements = $this->viewElementsDynameek($items, $langcode);
    }
    else {
      $elements = parent::viewElements($items, $langcode);

      // Replace file link by best quality URL if needed.
      $image_link_setting = $this->getSetting('image_link');
      if ($image_link_setting == 'file') {
        foreach ($elements as &$element) {
          if (!empty($element['#item']->_quality_url)) {
            $element['#url'] = $element['#item']->_quality_url;
          }
        }
      }
    }
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function deriveMediaDefaultNameFromUrl($url) {
    return KeepeekFormatterBase::deriveMediaDefaultNameFromUrl($url);
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    return KeepeekFormatterBase::isApplicable($field_definition);
  }

  /**
   * Build Dynameek URL based on the image style effects.
   *
   * @param string $url
   *   The asset URL.
   * @param \Drupal\image\ImageStyleInterface|null $image_style
   *   The image style.
   *
   * @return string
   *   The Dynameek URL.
   */
  protected function buildDynameekUrl($url, ImageStyleInterface $image_style = NULL) {
    // Leave original image for missing image style.
    if (empty($image_style)) {
      return $url;
    }
    $config = \Drupal::config('keepeek.settings');
    $key = hex2bin(trim($config->get('dynameek.key')));
    $domain = rtrim($config->get('dynameek.domain'), '/');

    $dynameek_effects = [];
    $dynameek_image_effect_manager = \Drupal::service('plugin.manager.keepeek.dynameek_image_effect');
    foreach ($image_style->getEffects() as $effect) {
      $plugin_id = $effect->getPluginId();
      $configuration = $effect->getConfiguration()['data'];
      if ($dynameek_image_effect_manager->hasDefinition($plugin_id)) {
        $dynameek_effects[] = $dynameek_image_effect_manager->createInstance($plugin_id, $configuration);
      }
    }

    $result = [];
    foreach ($dynameek_effects as $dynameek_effect) {
      // Merge all successfully transformed effects together.
      $effect_result = $dynameek_effect->transform($url, $this->json);
      if (!empty($effect_result)) {
        $result = array_replace_recursive($result, $effect_result);
      }
    }
    if (!empty($result)) {
      $action_url = '/' . $result['action'] . '?' . http_build_query($result['parameters']);
      // Create the hash hmac with the parameters.
      $hash = hash_hmac('sha256', $action_url, $key, TRUE);
      // Encode the hash hmac value with base 64 encode.
      $signature = str_replace('=', '', strtr(base64_encode($hash), '+/', '-_'));
      // Creating the final URL for rendition of Keepeek image.
      return $domain . '/' . $signature . $action_url;
    }

    return $url;
  }

}
