<?php

namespace Drupal\keepeek\Plugin\media\Source;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\keepeek\Event\KeepeekEvents;
use Drupal\keepeek\Event\KeepeekMediaNameResolveEvent;
use Drupal\keepeek\Plugin\Field\FieldFormatter\KeepeekFormatterBase;
use Drupal\keepeek\Service\KeepeekManager;
use Drupal\media\MediaInterface;
use Drupal\media\MediaSourceBase;
use Drupal\media\MediaSourceFieldConstraintsInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

const KEEPEEK_MEDIA_TYPE = 'keepeek_media_type';

/**
 * Keepeek media source.
 *
 * @MediaSource(
 *   id = "keepeek",
 *   label = @Translation("Keepeek"),
 *   description = @Translation("Keepeek media"),
 *   allowed_field_types = {"string", "string_long"},
 *   default_thumbnail_filename = "generic.png",
 *   forms = {
 *     "media_library_add" = "Drupal\keepeek\Form\KeepeekMediaForm"
 *   }
 * )
 */
class KeepeekSource extends MediaSourceBase implements MediaSourceFieldConstraintsInterface {

  /**
   * Key for "Name" metadata attribute.
   *
   * @var string
   */
  const METADATA_ATTRIBUTE_NAME = 'name';

  const METADATA_ATTRIBUTE_JSON = 'json';

  /**
   * @var EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * Constructs a new class instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   Entity field manager service.
   * @param \Drupal\Core\Field\FieldTypePluginManagerInterface $field_type_manager
   *   The field type plugin manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event_dispatcher service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, FieldTypePluginManagerInterface $field_type_manager, ConfigFactoryInterface $config_factory, EventDispatcherInterface $event_dispatcher) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $entity_field_manager, $field_type_manager, $config_factory);
    $this->eventDispatcher = $event_dispatcher;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('plugin.manager.field.field_type'),
      $container->get('config.factory'),
      $container->get('event_dispatcher')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getMetadataAttributes() {
    return [
      static::METADATA_ATTRIBUTE_NAME => $this->t('Name'),
      static::METADATA_ATTRIBUTE_JSON => 'JSON',
    ];
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form[KEEPEEK_MEDIA_TYPE] = [
      '#type' => 'checkboxes',
      '#title' => t('Choose the types of media that will be offered by the Keepicker'),
      '#default_value' => $this->configuration[KEEPEEK_MEDIA_TYPE] ?? [
        'PICTURE',
        'VIDEO',
        'DOCUMENT',
        'SOUND',
        'OTHER',
        'RICH_MEDIA',
      ],
      '#options' => [
        'PICTURE' => $this->t('Images'),
        'VIDEO' => $this->t('Videos'),
        'DOCUMENT' => $this->t('Documents'),
        'SOUND' => $this->t('Audios'),
        'OTHER' => $this->t('Other files'),
        'RICH_MEDIA' => $this->t('Rich media'),
      ],
      '#required' => TRUE,
    ];

    return $form;
  }

  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    $media_type = $form_state->getValue(KEEPEEK_MEDIA_TYPE);
    $this->configuration[KEEPEEK_MEDIA_TYPE] = $media_type;
  }

  public function addJsonFieldToConfiguration($type) {
    $field_name = parent::getSourceFieldName() . '_json';

    \Drupal::logger('keepeek')
      ->info('Creating JSON field %fieldName', ['%fieldName' => $field_name]);

    $this->entityTypeManager
      ->getStorage('field_storage_config')
      ->create([
        'entity_type' => 'media',
        'field_name' => $field_name,
        'type' => "string_long",
      ])->save();

    \Drupal::logger('keepeek')
      ->info('Creating JSON field configutation %fieldName inside bundle %bundle', [
        '%fieldName' => $field_name,
        '%bundle' => $type->id(),
      ]);

    FieldConfig::create([
      'bundle' => $type->id(),
      'entity_type' => 'media',
      'field_name' => $field_name,
      'label' => $field_name,
      'description' => '',
      'required' => FALSE,
      'default_value' => [],
      'default_value_callback' => '',
      'translatable' => FALSE,
      'settings' => [
        'max_length' => 200000,
        'case_sensitive' => FALSE,
        'is_ascii' => FALSE,
      ],
    ])->save();

    return $field_name;
  }

  /**
   * {@inheritdoc}
   */
  public function getMetadata(MediaInterface $media, $attribute_name) {
    $url = $media->get($this->configuration['source_field'])->value;
    // If the source field is not required, it may be empty.
    if (empty($url)) {
      return parent::getMetadata($media, $attribute_name);
    }
    switch ($attribute_name) {
      case static::METADATA_ATTRIBUTE_NAME:
      case 'default_name':
        // Use asset title as the default name if available.
        $data = KeepeekManager::getDataFromMedia($media);
        if (!empty($data['title']['value'])) {
          $default_name = strlen($data['title']['value']) > 255
            ? substr($data['title']['value'], 0, 252) . '...'
            : $data['title']['value'];
        } else {
          $default_name = KeepeekFormatterBase::deriveMediaDefaultNameFromUrl($url);
        }

        // Allow other modules to alter resolved name based on their needs.
        $alterNameEvent = new KeepeekMediaNameResolveEvent($default_name, $this->configuration, $media, $data);
        $this->eventDispatcher->dispatch($alterNameEvent, KeepeekEvents::KEEPEEK_DEFAULT_NAME_ALTER);
        return $alterNameEvent->getName();

      case 'thumbnail_uri':
        return $url;

      default:
        return parent::getMetadata($media, $attribute_name);
    }
  }

  /**
   * Figure out the formatter class to be used on a given media entity.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity we are interested in.
   *
   * @return string
   *   The FQN of the formatter class configured in the `default` media display
   *   for the media source field.
   */
  public function getFormatterClass(MediaInterface $media) {
    $field_definition = $this->getSourceFieldDefinition($media->bundle->entity);

    // @todo There is probably a better way for this class to figure out what
    // formatter class is being used.
    $display = EntityViewDisplay::load('media.' . $media->bundle() . '.default');
    $components = $display->getComponents();
    $formatter_config = $components[$field_definition->getName()] ?? [];
    if (empty($formatter_config['settings']['formatter_class'])) {
      throw new \LogicException('The Keepeek validator needs the _default_ media display to be configured, and for the source field to use any of the formatters provided by the Keepeek module.');
    }

    // See KeepeekFormatterBase::defaultSettings() for where this is
    // defined/enforced.
    return $formatter_config['settings']['formatter_class'];
  }

  /**
   * {@inheritdoc}
   */
  public function getSourceFieldConstraints() {
    return [
      'keepeek' => [],
    ];
  }

}
