<?php

namespace Drupal\keepeek\Service;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\media\MediaInterface;

/**
 * KeepeekManager
 */
class KeepeekManager {

  /**
   * @var string
   */
  public const PERMALINK_TYPE_ORIGINAL = 'original';

  public const PERMALINK_TYPE_PREVIEW = 'preview';

  public const PERMALINK_TYPE_MEDIUM = 'medium';

  public const PERMALINK_TYPE_XLARGE = 'xlarge';

  public const PERMALINK_TYPE_WHR = 'whr';

  public const PERMALINK_TYPE_XLARGE_MAX_SIZE = 448;

  /**
   * @var string
   */
  protected const TYPE_NORMAL = 'type-normal';

  protected const TYPE_RESIZE = 'type-resize';

  /**
   * Keepeek permalink type in Keepeek URL
   */
  const KEEPEEK_PERMALINK_TYPE_PATTERN = '-(' . self::PERMALINK_TYPE_WHR . '|' . self::PERMALINK_TYPE_PREVIEW . ')\.jpg';

  /**
   * Conversion pattern to display lower resolution than WHR for thumbnails
   */
  const DRUPAL_IMAGE_STYLE_CONVERSION_MEDIA_TYPE_PATTERN = '-' . self:: PERMALINK_TYPE_XLARGE . '.jpg';

  /**
   * Keepeek extension pattern from WHR format or Dynameek
   */
  const KEEPEEK_EXTENSION_PATTERN = '(jpg|jpeg|png|webp|heif)';

  /**
   * Drupal image style conversion extension pattern
   */
  const DRUPAL_IMAGE_STYLE_CONVERSION_EXTENSION_PATTERN = '(jpg|jpeg|jpe|png|gif|webp)';

  /**
   * This method is used to correct the Drupal URL path.
   *
   * It takes a Drupal URL path as a string, checks if it contains a URL, and
   * if so, it decodes the URL, removes the 'itok' query parameter if it
   * exists, checks for double extensions, and returns the corrected URL.
   *
   * - If the URL contains the 'itok' query parameter, it is removed.
   * - If the URL contains two extensions (e.g., ".jpg.webp"), the second
   * extension is removed to avoid a 404 error.
   *
   * @param string $path Original Drupal URL path.
   *
   * @return string Asset URL.
   */
  public static function getCorrectUrl(string $url, int $width = 0, int $height = 0): string {
    $path_array = explode('https/', $url);
    if (count($path_array) > 1) {
      $url = urldecode($path_array[1]);
      $url = sprintf('https://%s', $url);

      // Remove itok for external urls
      $itok_pos = strpos($url, '?itok=');
      if ($itok_pos !== FALSE) {
        $url = substr($url, 0, $itok_pos);
      }

      // Switch to xlarge for small pictures when loading whr
      if (
        $width < self::PERMALINK_TYPE_XLARGE_MAX_SIZE && $width > 0
        && $height < self::PERMALINK_TYPE_XLARGE_MAX_SIZE && $height > 0
        && preg_match('/' . self::KEEPEEK_PERMALINK_TYPE_PATTERN . '/', $url)
      ) {
        $url = preg_replace(
          '/(.*)' . self::KEEPEEK_PERMALINK_TYPE_PATTERN . '(.*)$/',
           '$1' . self::DRUPAL_IMAGE_STYLE_CONVERSION_MEDIA_TYPE_PATTERN . '$3', 
           $url);
      }

      // Detect if the URL contains a double extension where the first extension originates from the Keepeek image,
      // and the second extension is added by Drupal through an "image style" effect.
      // This occurs, for instance, when Drupal applies a format conversion (e.g., to WebP) on thumbnails in the media library.
      // The double extension should have first an available Keepeek extension
      // and the second an available in Drupal image style conversion extension.
      if (preg_match('/\.' . self::KEEPEEK_EXTENSION_PATTERN . '\.' . self::DRUPAL_IMAGE_STYLE_CONVERSION_EXTENSION_PATTERN . '$/', $url)) {
        // If a double extension is detected, remove the second extension to avoid a 404 error.
        $url = preg_replace('/\.' . self::DRUPAL_IMAGE_STYLE_CONVERSION_EXTENSION_PATTERN . '$/', '', $url);
      }
    }
    return $url;
  }

  /**
   * Extracts data from a field item.
   *
   * @param FieldItemInterface $item The field item to extract data from.
   *
   * @return array The extracted data.
   */
  public static function getDataFromFieldItem(FieldItemInterface $item): array {
    $media = $item->getParent()->getParent()->getValue();
    $field_name = self::getJsonFieldName($media);

    $data = [];
    if ($item->getParent() instanceof FieldItemListInterface
      && $media instanceof MediaInterface
      && $media->hasField($field_name)
    ) {
      $data = self::getDataFromMedia($media);
    }
    return $data;
  }

  /**
   * Extracts data from a media item.
   *
   * @param MediaInterface $media The media item to extract data from.
   *
   * @return array The extracted data.
   */
  public static function getDataFromMedia(MediaInterface $media): array {
    $field_name = self::getJsonFieldName($media);

    $data = [];
    if ($media->hasField($field_name)) {
      foreach ($media->get($field_name)->getValue() as $media_field_value) {
        $data = json_decode($media_field_value['value'], TRUE) ?? [];
      }
    }
    return $data;
  }

  /**
   * Gets the permalink URL of a given type from the provided data.
   *
   * @param array $data The data to extract the permalink URL from.
   * @param string $type The type of permalink URL to get.
   *
   * @return string The permalink URL.
   */
  public static function getPermalinkUrl(array $data, string $type = self::PERMALINK_TYPE_ORIGINAL): string {
    $url = '';
    if (isset($data['permalinks'])) {
      foreach ($data['permalinks'] as $permalink) {
        if (isset($permalink['title']) && $permalink['title'] === $type) {
          $url = $permalink['url'];
          break;
        }
      }
    }
    return $url;
  }

  /**
   * Checks if the provided data represents a video.
   *
   * @param array $data The data to check.
   *
   * @return bool True if the data represents a video, false otherwise.
   */
  public static function isVideo(array $data): bool {
    return str_contains($data['mediaType'], 'video');
  }

  /**
   * Extracts the filename from a URI.
   *
   * @param string $uri The URI to extract the filename from.
   *
   * @return string The extracted filename.
   */
  public static function getFilenameFromUri(string $uri): string {
    if (self::getUriType($uri) === self::TYPE_RESIZE) {
      $uri_values = UrlHelper::parse($uri);
      $uri = $uri_values['query']['src'] ?? $uri;
    }
    return basename($uri);
  }

  /**
   * Gets the name of the JSON field from a media item.
   *
   * @param MediaInterface $media The media item to get the JSON field name
   *   from.
   *
   * @return string The name of the JSON field.
   */
  public static function getJsonFieldName(MediaInterface $media) {
    $field_map = $media->bundle->entity->getFieldMap();

    foreach ($field_map as $field_name => $field_label) {
      if ($field_name == 'json') {
        return $field_label;
      }
    }
  }

  /**
   * Determines the type of a URI.
   *
   * @param string $uri The URI to determine the type of.
   *
   * @return string The type of the URI.
   */
  public static function getUriType(string $uri): string {
    return str_contains($uri, '/resize') || str_contains($uri, '/crop') ? self::TYPE_RESIZE : self::TYPE_NORMAL;
  }

}
