<?php

namespace Drupal\keepeek\Plugin\Field\FieldFormatter;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\image\Entity\ImageStyle;
use Drupal\image\Plugin\Field\FieldFormatter\ImageFormatter;
use Drupal\keepeek\KeepeekFormatterInterface;

/**
 * Plugin implementation of the 'Keepeek Image' formatter.
 *
 * @FieldFormatter(
 *   id = "keepeek_image",
 *   label = @Translation("Keepeek Image"),
 *   field_types = {
 *     "string"
 *   }
 * )
 */
class KeepeekImageFormatter extends ImageFormatter implements KeepeekFormatterInterface {

  use KeepeekImageFormatterTrait;

  /**
   * Builds a renderable array for a field value using Dynameek.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The field values to be rendered.
   * @param string $langcode
   *   The language that should be used to render the field.
   *
   * @return array
   *   A renderable array for $items, as an array of child elements keyed by
   *   consecutive numeric indexes starting from 0.
   */
  public function viewElementsDynameek(FieldItemListInterface $items, $langcode) {
    $elements = [];

    // Early opt-out if the field is empty.
    if ($items->isEmpty()) {
      return $elements;
    }
    $entity = $items->getEntity();

    $image_link_setting = $this->getSetting('image_link');
    $image_style_setting = $this->getSetting('image_style');
    $image_loading_settings = $this->getSetting('image_loading');

    $url = NULL;
    // Check if the formatter involves a link.
    if ($image_link_setting == 'content') {
      if (!$entity->isNew()) {
        $url = $entity->toUrl()->toString();
      }
    }
    elseif ($image_link_setting == 'file') {
      $link_file = TRUE;
    }

    // Collect cache tags to be added for each item in the field.
    $cache_tags = [];
    $image_style = ImageStyle::load($image_style_setting);
    if (!empty($image_style)) {
      $cache_tags = $image_style->getCacheTags();
    }
    $cache_tags = Cache::mergeTags($cache_tags, $entity->getCacheTags());

    // Process every field item separately.
    foreach ($items as $delta => $item) {
      $quality_url = _keepeek_quality_url($item->value, $this->json);
      // Link image element to the original file.
      if (isset($link_file)) {
        $url = $quality_url;
      }
      $attributes = $item->_attributes;
      $attributes['loading'] = $image_loading_settings['attribute'];
      $attributes['alt'] = NULL;

      // Build render element.
      $elements[$delta] = [
        '#theme' => 'image',
        '#uri' => $this->buildDynameekUrl($quality_url, $image_style),
        '#attributes' => $attributes,
        '#style_name' => $image_style_setting,
        '#cache' => [
          'tags' => $cache_tags,
        ],
      ];
      if (!empty($url)) {
        $elements[$delta]['#prefix'] = '<a href="' . $url . '">';
        $elements[$delta]['#suffix'] = '</a>';
      }
    }

    return $elements;
  }

}
