<?php

namespace Drupal\keepeek\Plugin\Field\FieldFormatter;

use Drupal\breakpoint\BreakpointInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Template\Attribute;
use Drupal\image\Entity\ImageStyle;
use Drupal\keepeek\KeepeekFormatterInterface;
use Drupal\responsive_image\Entity\ResponsiveImageStyle;
use Drupal\responsive_image\Plugin\Field\FieldFormatter\ResponsiveImageFormatter;

/**
 * Plugin implementation of the 'Keepeek Responsive Image' formatter.
 *
 * @FieldFormatter(
 *   id = "keepeek_responsive_image",
 *   label = @Translation("Keepeek Responsive Image"),
 *   field_types = {
 *     "string"
 *   }
 * )
 */
class KeepeekResponsiveImageFormatter extends ResponsiveImageFormatter implements KeepeekFormatterInterface {

  use KeepeekImageFormatterTrait;

  /**
   * Builds a renderable array for a field value using Dynameek.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The field values to be rendered.
   * @param string $langcode
   *   The language that should be used to render the field.
   *
   * @return array
   *   A renderable array for $items, as an array of child elements keyed by
   *   consecutive numeric indexes starting from 0.
   */
  public function viewElementsDynameek(FieldItemListInterface $items, $langcode) {
    $elements = [];

    // Early opt-out if the field is empty.
    if ($items->isEmpty()) {
      return $elements;
    }
    $entity = $items->getEntity();

    $image_link_setting = $this->getSetting('image_link');
    $image_style_setting = $this->getSetting('responsive_image_style');
    $image_loading_settings = $this->getSetting('image_loading');

    $url = NULL;
    // Check if the formatter involves a link.
    if ($image_link_setting == 'content') {
      if (!$entity->isNew()) {
        $url = $entity->toUrl()->toString();
      }
    }
    elseif ($image_link_setting == 'file') {
      $link_file = TRUE;
    }

    // Collect cache tags to be added for each item in the field.
    $cache_tags = [];
    $responsive_image_style = ResponsiveImageStyle::load($image_style_setting);
    $image_style_ids = [];
    if (!empty($responsive_image_style)) {
      $cache_tags = $responsive_image_style->getCacheTags();
      $image_style_ids = $responsive_image_style->getImageStyleIds();
    }
    $image_styles = ImageStyle::loadMultiple($image_style_ids);
    foreach ($image_styles as $image_style) {
      $cache_tags = Cache::mergeTags($cache_tags, $image_style->getCacheTags());
    }

    // Process every field item separately.
    foreach ($items as $delta => $item) {
      $quality_url = _keepeek_quality_url($item->value, $this->json);
      // Link image element to the original file.
      if (isset($link_file)) {
        $url = $quality_url;
      }
      $attributes = $item->_attributes;
      $attributes['loading'] = $image_loading_settings['attribute'];
      $attributes['alt'] = NULL;

      // Build render element.
      $elements[$delta] = [
        '#theme' => 'keepeek_responsive_image',
        '#sources' => [],
        '#cache' => [
          'tags' => $cache_tags,
        ],
      ];
      if (!empty($url)) {
        $elements[$delta]['#url'] = $url;
      }

      // Prepare array with the asset URL and dimensions.
      $variables = ['url' => $quality_url, 'width' => NULL, 'height' => NULL];
      if (!empty($this->json->fields)) {
        foreach ($this->json->fields as $field) {
          if (in_array($field->id, ['width', 'height'])) {
            $variables[$field->id] = intval($field->value);
          }
        }
      }

      // Retrieve all breakpoints and multipliers in reverse order.
      $breakpoints = array_reverse(\Drupal::service('breakpoint.manager')
        ->getBreakpointsByGroup($responsive_image_style->getBreakpointGroup()));
      foreach ($responsive_image_style->getKeyedImageStyleMappings() as $breakpoint_id => $multipliers) {
        if (isset($breakpoints[$breakpoint_id])) {
          $elements[$delta]['#sources'][] = $this->buildSourceAttributes($variables, $breakpoints[$breakpoint_id], $multipliers, $image_styles);
        }
      }

      // Use image tag for only one source tag with an empty media attribute.
      if (count($elements[$delta]['#sources']) === 1 && !isset($elements[$delta]['#sources'][0]['media'])) {
        $elements[$delta]['#output_image_tag'] = TRUE;
        foreach ($elements[$delta]['#sources'][0] as $attribute => $value) {
          if ($attribute != 'type') {
            $attributes[$attribute] = $value;
          }
        }
      }
      // Prepare the fallback image with the src attribute.
      $elements[$delta]['#img_element'] = [
        '#theme' => 'image',
        '#uri' => $this->buildDynameekUrl($quality_url, $image_styles[$responsive_image_style->getFallbackImageStyle()]),
        '#attributes' => [],
      ];

      // Prevent recalculation of layout on image load.
      $dimensions = responsive_image_get_image_dimensions($responsive_image_style->getFallbackImageStyle(), ['width' => $variables['width'], 'height' => $variables['height']], NULL);
      $elements[$delta]['#img_element']['#width'] = $dimensions['width'];
      $elements[$delta]['#img_element']['#height'] = $dimensions['height'];

      if (isset($attributes['alt'])) {
        $elements[$delta]['#img_element']['#alt'] = $attributes['alt'];
        unset($attributes['alt']);
      }
      if (isset($attributes['title'])) {
        $elements[$delta]['#img_element']['#title'] = $attributes['title'];
        unset($attributes['title']);
      }
      if (isset($elements[$delta]['#img_element']['#width'])) {
        $attributes['width'] = $elements[$delta]['#img_element']['#width'];
      }
      if (isset($elements[$delta]['#img_element']['#height'])) {
        $attributes['height'] = $elements[$delta]['#img_element']['#height'];
      }
      $elements[$delta]['#img_element']['#attributes'] = $attributes;
    }

    return $elements;
  }

  /**
   * Builds attributes for <source> tags to be used in a <picture> tag.
   *
   * @param array $variables
   *   An array with the asset URL and dimensions.
   * @param \Drupal\breakpoint\BreakpointInterface $breakpoint
   *   The breakpoint for this source tag.
   * @param array $multipliers
   *   An array with multipliers as keys and image style mappings as values.
   *
   * @return \Drupal\Core\Template\Attribute
   *   An object of attributes for the source tag.
   */
  protected function buildSourceAttributes(array $variables, BreakpointInterface $breakpoint, array $multipliers, array $image_styles) {
    $extension = pathinfo($variables['url'], PATHINFO_EXTENSION);
    $sizes = [];
    $srcset = [];
    $derivative_mime_types = [];
    // Traverse multipliers in reverse so the largest image is processed last.
    foreach (array_reverse($multipliers) as $multiplier => $image_style_mapping) {
      switch ($image_style_mapping['image_mapping_type']) {
        case 'sizes':
          // Loop through the image styles for this breakpoint and multiplier.
          foreach ($image_style_mapping['image_mapping']['sizes_image_styles'] as $image_style_name) {
            // Get the dimensions.
            $dimensions = responsive_image_get_image_dimensions($image_style_name, ['width' => $variables['width'], 'height' => $variables['height']], NULL);
            // Get MIME type.
            $derivative_mime_type = responsive_image_get_mime_type($image_style_name, $extension);
            $derivative_mime_types[] = $derivative_mime_type;
            // Add the image source with its multiplier.
            $image_style = $image_styles[$image_style_name];

            // Skip empty width when width descriptor is used in a srcset.
            if (is_null($dimensions['width'])) {
              throw new \LogicException("Could not determine image width for '{$variables['url']}' using image style with ID: $image_style_name. This image style can not be used for a responsive image style mapping using the 'sizes' attribute.");
            }
            // Use the image width as key so we can sort the array later on.
            $srcset[intval($dimensions['width'])] = $this->buildDynameekUrl($variables['url'], $image_style) . ' ' . $dimensions['width'] . 'w';
            $sizes = array_merge(explode(',', $image_style_mapping['image_mapping']['sizes']), $sizes);
          }
          break;
        case 'image_style':
          // Get MIME type.
          $derivative_mime_type = responsive_image_get_mime_type($image_style_mapping['image_mapping'], $extension);
          $derivative_mime_types[] = $derivative_mime_type;
          // Add the image source with its multiplier.
          $image_style = $image_styles[$image_style_mapping['image_mapping']];
          $srcset[intval(mb_substr($multiplier, 0, -1) * 100)] = $this->buildDynameekUrl($variables['url'], $image_style) . ' ' . $multiplier;
          $dimensions = responsive_image_get_image_dimensions($image_style_mapping['image_mapping'], ['width' => $variables['width'], 'height' => $variables['height']], NULL);
          break;
      }
    }
    // Sort the srcset from small to large image width or multiplier.
    ksort($srcset);
    $source_attributes = new Attribute([
      'srcset' => implode(', ', array_unique($srcset)),
    ]);
    $media_query = trim($breakpoint->getMediaQuery());
    if (!empty($media_query)) {
      $source_attributes->setAttribute('media', $media_query);
    }
    if (count(array_unique($derivative_mime_types)) == 1) {
      $source_attributes->setAttribute('type', $derivative_mime_types[0]);
    }
    if (!empty($sizes)) {
      $source_attributes->setAttribute('sizes', implode(',', array_unique($sizes)));
    }
    // The srcset attribute images should all have the same aspect ratio.
    if (!empty($dimensions['width']) && !empty($dimensions['height'])) {
      $source_attributes->setAttribute('width', $dimensions['width']);
      $source_attributes->setAttribute('height', $dimensions['height']);
    }
    return $source_attributes;
  }

}
