<?php

namespace Drupal\Tests\keepeek\Kernel;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\keepeek\Controller\KeepeekEndpoint;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;
use Firebase\JWT\JWT;
use Symfony\Component\HttpFoundation\Request;

/**
 * Test Keepeek endpoints.
 *
 * @covers \Drupal\keepeek\Controller\KeepeekEndpoint
 * @group keepeek
 */
class KeepeekEndpointTest extends KernelTestBase {

  use MediaTypeCreationTrait;

  /**
   * The endpoint controller.
   *
   * @var \Drupal\keepeek\Controller\KeepeekEndpoint
   */
  protected $controller;

  /**
   * The endpoint controller getMediaIds method.
   *
   * @var \ReflectionMethod
   */
  protected $getMediaIdsMethod;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'field',
    'image',
    'keepeek',
    'media',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('media');

    $this->createMediaType('keepeek', [
      'id' => 'keepeek',
      'field_map' => ['json' => 'field_media_keepeek_1_json'],
    ]);
    FieldStorageConfig::create([
      'field_name' => 'field_media_keepeek_1_json',
      'entity_type' => 'media',
      'type' => 'string_long',
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_media_keepeek_1_json',
      'entity_type' => 'media',
      'bundle' => 'keepeek',
    ])->save();

    $this->controller = $this->container->get('class_resolver')
      ->getInstanceFromDefinition(KeepeekEndpoint::class);

    $reflection_class = new \ReflectionClass($this->controller);
    $this->getMediaIdsMethod = $reflection_class->getMethod('getMediaIds');
  }

  /**
   * Tests that correct exception is thrown when JWT key is not configured.
   */
  public function testExceptionEmptyKey() {
    $request = new Request();

    $this->expectExceptionMessage('The endpoint JWT key is not configured.');
    $this->getMediaIdsMethod->invokeArgs($this->controller, [$request]);
  }

  /**
   * Tests that correct exception is thrown when JWT token is missing.
   */
  public function testExceptionEmptyToken() {
    $request = new Request();
    $key = 'test_key';
    $this->config('keepeek.settings')->set('jwt_key', $key)->save();

    $this->expectExceptionMessage('The request JWT token is missing.');
    $this->getMediaIdsMethod->invokeArgs($this->controller, [$request]);
  }

  /**
   * Tests that correct exception is thrown when JWT token is incorrect.
   */
  public function testExceptionIncorrectToken() {
    $request = new Request();
    $key = 'test_key';
    $this->config('keepeek.settings')->set('jwt_key', $key)->save();
    $request->headers->set('Authorization', 'Bearer incorrect_token');

    $this->expectExceptionMessageMatches('/^The JWT validation problem:/');
    $this->getMediaIdsMethod->invokeArgs($this->controller, [$request]);
  }

  /**
   * Tests that media IDs from the request are extracted correctly.
   */
  public function testMediaIds() {
    $request = new Request();
    $key = 'test_key';
    $this->config('keepeek.settings')->set('jwt_key', $key)->save();
    $payload = ['asset_ids' => [12345]];
    $authorization = JWT::encode($payload, $key, 'HS256');
    $request->headers->set('Authorization', $authorization);

    $ids = $this->getMediaIdsMethod->invokeArgs($this->controller, [$request]);
    $this->assertEmpty($ids);
  }

}
