<?php

namespace Drupal\keepeek\Form;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Site\Settings;

/**
 * Defines a form for configuring the Keepeek module.
 */
class KeepeekSettingsForm extends ConfigFormBase {

  /**
   * @var array
   */
  protected array $formItems = [
    'api_endpoint' => [
      'type' => 'url',
      'attributes' => ['placeholder' => 'https://'],
      'description' => 'Keepeek media library URL (ie. <code>https://www.keepeek.com</code>)',
    ],
    'keycloak_client_id' => ['type' => 'textfield', 'required' => FALSE],
    'keycloak_idp' => ['type' => 'textfield', 'required' => FALSE],
    'data_locale' => [
      'type' => 'textfield',
      'required' => FALSE,
      'description' => 'Two chars format (ie. <code>FR</code>)',
    ],
    'ui_locale' => [
      'type' => 'textfield',
      'required' => FALSE,
      'description' => 'Two chars format (ie. <code>FR</code>)',
    ],
  ];

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames() {
    return ['keepeek.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'keepeek_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('keepeek.settings');

    $form['configuration'] = [
      '#type' => 'details',
      '#title' => $this->t('Keepeek configuration'),
      '#open' => TRUE,
    ];

    foreach ($this->formItems as $item_name => $item_values) {
      $form['configuration'][$item_name] = [
        '#type' => $item_values['type'],
        '#title' => $this->t($item_name),
        '#default_value' => $config->get($item_name),
        '#required' => $item_values['required'] ?? TRUE,
        '#description' => $item_values['description'] ?? NULL,
        '#attributes' => $item_values['attributes'] ?? [],
      ];
    }

    $jwt_key = $config->get('jwt_key');
    if (empty($jwt_key)) {
      $jwt_key = $this->generateJwtKey();
    }
    $form['endpoint'] = [
      '#type' => 'fieldset',
    ];
    $form['endpoint']['jwt_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('JWT key'),
      '#description' => $this->t('Use this key for communication with the Keepeek endpoints.'),
      '#default_value' => $jwt_key,
      '#required' => TRUE,
    ];
    $form['endpoint']['jwt_key_refresh'] = [
      '#type' => 'submit',
      '#name' => 'jwt_key_refresh',
      '#value' => $this->t('Refresh JWT key'),
    ];

    $form['dynameek'] = [
      '#type' => 'details',
      '#title' => $this->t('Dynameek options'),
      '#tree' => TRUE,
    ];
    $form['dynameek']['key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Dynameek key'),
      '#description' => $this->t('Use this key for communication with the Dynameek service.'),
      '#default_value' => $config->get('dynameek.key'),
    ];
    $form['dynameek']['domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Dynameek domain'),
      '#description' => $this->t('Use this domain for Dynameek images.'),
      '#default_value' => $config->get('dynameek.domain'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $settings = $this->config('keepeek.settings');

    foreach ($this->formItems as $item_name => $item_values) {
      $settings->set($item_name, $form_state->getValue($item_name));
    }

    // Refresh JWT key if requested.
    $element = $form_state->getTriggeringElement();
    if ($element['#name'] === 'jwt_key_refresh') {
      $settings->set('jwt_key', $this->generateJwtKey());
    }
    else {
      $settings->set('jwt_key', $form_state->getValue('jwt_key'));
    }

    $dynameek = $form_state->getValue('dynameek');
    $settings->set('dynameek.key', $dynameek['key']);
    $settings->set('dynameek.domain', $dynameek['domain']);

    $settings->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Generate JWT key.
   *
   * @return string
   *   The generated JWT key.
   */
  protected function generateJwtKey() {
    return Crypt::hmacBase64(time(), Settings::getHashSalt());
  }

}
