<?php

namespace Drupal\keepeek\Tests;

use Drupal\keepeek\Service\KeepeekManager;
use PHPUnit\Framework\TestCase;

class KeepeekManagerTest extends TestCase {

  public function testGetCorrectUrlWithInnerUrlAndItokParam() {
    $url = 'https://google.com/https/keepeek.com/path/to/picture.png?itok=12345';
    $expected = 'https://keepeek.com/path/to/picture.png';
    $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url));
  }

  public function testGetCorrectUrlWithInnerUrl() {
    $url = 'https://google.com/https/keepeek.com/path/to/picture.png';
    $expected = 'https://keepeek.com/path/to/picture.png';
    $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url));
  }

  public function testGetCorrectUrlWithoutInnerUrl() {
    $url = 'https://keepeek.com/path/to/picture.png';
    $expected = 'https://keepeek.com/path/to/picture.png';
    $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url));
  }

  public function testGetCorrectUrlWhrAsPng() {
    $url = 'https://keepeek.com/path/to/picture-whr.png';
    $expected = 'https://keepeek.com/path/to/picture-whr.png';
    $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url));
  }

  public function testGetCorrectUrlWhrNoTransformations() {
    $url = 'https://keepeek.com/path/to/picture-whr.jpg';
    $expected = 'https://keepeek.com/path/to/picture-whr.jpg';
    $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url));
  }

  public function testGetCorrectUrlWhrNotAThumbnail() {
    $url = 'https://keepeek.com/path/to/picture-whr.jpg';
    $expected = 'https://keepeek.com/path/to/picture-whr.jpg';
    $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url, 100, 100));
  }

  public function testGetCorrectUrlWhrAsWhrThumbnail() {
    $url = 'http://drupal.local/sites/default/files/styles/thumbnail/https/keepeek.com/pm_123_456_789-abcdef-whr.jpg.webp?itok=123456';
    $expected = 'https://keepeek.com/pm_123_456_789-abcdef-whr.jpg';
    $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url, 1000, 1000));
  }

  public function testGetCorrectUrlWhrAsXlargeThumbnail() {
    $url = 'http://drupal.local/sites/default/files/styles/thumbnail/https/keepeek.com/pm_123_456_789-abcdef-whr.jpg.webp?itok=123456';
    $expected = 'https://keepeek.com/pm_123_456_789-abcdef-xlarge.jpg';
    $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url, 100, 100));
  }

  public function testKeepWhrIfSizeIsUndefined() {
  $url = 'http://drupal.local/sites/default/files/styles/thumbnail/https/keepeek.com/pm_123_456_789-abcdef-whr.jpg.webp?itok=123456';
  $expected = 'https://keepeek.com/pm_123_456_789-abcdef-whr.jpg';
  $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url, -1, -1));
}

public function testKeepWhrIfOnlyHeightIsLarge() {
  $url = 'http://drupal.local/sites/default/files/styles/thumbnail/https/keepeek.com/pm_123_456_789-abcdef-whr.jpg.webp?itok=123456';
  $expected = 'https://keepeek.com/pm_123_456_789-abcdef-whr.jpg';
  $this->assertEquals($expected, KeepeekManager::getCorrectUrl($url, 100, 1000));
}

  public function testTypeNormal() {
    $uri = 'https://keepeek.com/path/to/picture.png';
    $expected = 'type-normal';
    $this->assertEquals($expected, KeepeekManager::getUriType($uri));
  }

  public function testTypeCrop() {
    $uri = 'https://keepeek.com/path/to/crop/picture.png';
    $expected = 'type-resize';
    $this->assertEquals($expected, KeepeekManager::getUriType($uri));
  }

  public function testTypeResize() {
    $uri = 'https://keepeek.com/path/to/resize/picture.png';
    $expected = 'type-resize';
    $this->assertEquals($expected, KeepeekManager::getUriType($uri));
  }

  public function testGetFilenameFromResizeUri() {
    $uri = 'https://keepeek.com/path/to/resize/picture.png';
    $expected = 'picture.png';
    $this->assertEquals($expected, KeepeekManager::getFilenameFromUri($uri));
  }

  public function testGetFilenameFromUri() {
    $uri = 'https://keepeek.com/path/to/picture.png';
    $expected = 'picture.png';
    $this->assertEquals($expected, KeepeekManager::getFilenameFromUri($uri));
  }

  public function testGetPermalinkUrlWithMediumType() {
    $data = json_decode('{"permalinks": [{"title": "preview", "url": "https://keepeek.com/preview.jpg"},{"title": "medium", "url": "https://keepeek.com/medium.jpg"}]}', TRUE);
    $expected = 'https://keepeek.com/medium.jpg';
    $this->assertEquals($expected, KeepeekManager::getPermalinkUrl($data, KeepeekManager::PERMALINK_TYPE_MEDIUM));
  }

  public function testGetPermalinkUrlWithPreviewType() {
    $data = json_decode('{"permalinks": [{"title": "preview", "url": "https://keepeek.com/preview.jpg"},{"title": "medium", "url": "https://keepeek.com/medium.jpg"}]}', TRUE);
    $expected = 'https://keepeek.com/preview.jpg';
    $this->assertEquals($expected, KeepeekManager::getPermalinkUrl($data, KeepeekManager::PERMALINK_TYPE_PREVIEW));
  }

  public function testGetPermalinkUrlWithXlargeType() {
    $data = json_decode('{"permalinks": [{"title": "preview", "url": "https://keepeek.com/preview.jpg"},{"title": "medium", "url": "https://keepeek.com/medium.jpg"}]}', TRUE);
    $expected = '';
    $this->assertEquals($expected, KeepeekManager::getPermalinkUrl($data, KeepeekManager::PERMALINK_TYPE_XLARGE));
  }

  public function testIsVideoWithVideoUrl() {
    $data = json_decode('{"mediaType": "video/mp4"}', TRUE);
    $this->assertTrue(KeepeekManager::isVideo($data));
  }

  public function testIsVideoWithImageUrl() {
    $data = json_decode('{"mediaType": "image/png"}', TRUE);
    $this->assertFalse(KeepeekManager::isVideo($data));
  }

}
