<?php

namespace Drupal\Tests\keepeek\Kernel;

use Drupal\Core\Entity\Plugin\DataType\EntityAdapter;
use Drupal\Core\Field\FieldDefinition;
use Drupal\Core\Field\FieldItemList;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\keepeek\Plugin\Field\FieldFormatter\KeepeekImageFormatter;
use Drupal\keepeek\Plugin\Field\FieldFormatter\KeepeekResponsiveImageFormatter;
use Drupal\KernelTests\KernelTestBase;
use Drupal\media\Entity\Media;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;

/**
 * Test Keepeek formatters.
 *
 * @covers \Drupal\keepeek\Plugin\Field\FieldFormatter\KeepeekImageFormatter
 * @covers \Drupal\keepeek\Plugin\Field\FieldFormatter\KeepeekResponsiveImageFormatter
 * @group keepeek
 */
class KeepeekFormatterTest extends KernelTestBase {

  use MediaTypeCreationTrait;

  /**
   * The test items.
   *
   * @var \Drupal\Core\Field\FieldItemListInterface
   */
  protected $items;

  /**
   * The formatter configuration.
   *
   * @var array
   */
  protected $formatterConfiguration;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'breakpoint',
    'field',
    'file',
    'image',
    'keepeek',
    'media',
    'responsive_image',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->setInstallProfile('standard');
    $this->installConfig([
      'image',
      'keepeek',
    ]);

    $this->createMediaType('keepeek', [
      'id' => 'keepeek',
      'field_map' => ['json' => 'field_media_keepeek_1_json'],
    ]);
    FieldStorageConfig::create([
      'field_name' => 'field_media_keepeek_1_json',
      'entity_type' => 'media',
      'type' => 'string_long',
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_media_keepeek_1_json',
      'entity_type' => 'media',
      'bundle' => 'keepeek',
    ])->save();

    $definition = FieldConfig::load('media.keepeek.field_media_keepeek');
    $media = Media::create([
      'bundle' => 'keepeek',
      'field_media_keepeek_1_json' => '{"id":15868,"imagesServicePermalinks":"kpkp://iconeek/12799/15/15868-w2s35vsrsw.jpg"}',
    ]);
    $parent = EntityAdapter::createFromEntity($media);
    $this->items = FieldItemList::createInstance($definition, 'keepeek', $parent);
    $this->items->setValue([
      ['value' => 'https://assets-iconeek.keepeek.com/pm_12799_15_15886-3he1dvuc8i-whr.jpg'],
    ]);

    $this->formatterConfiguration = [
      'field_definition' => FieldDefinition::create('string'),
      'settings' => [
        'image_link' => '',
        'image_loading' => ['attribute' => 'eager'],
      ],
      'label' => 'visually_hidden',
      'view_mode' => 'default',
      'third_party_settings' => [],
    ];
  }

  /**
   * Tests that image formatter is processed correctly.
   */
  public function testImage() {
    $configuration = $this->formatterConfiguration;
    $configuration['settings']['image_style'] = 'medium';
    $configuration['settings']['formatter_class'] = KeepeekImageFormatter::class;

    /** @var \Drupal\keepeek\Plugin\Field\FieldFormatter\KeepeekImageFormatter $formatter */
    $formatter = $this->container->get('plugin.manager.field.formatter')
      ->createInstance('keepeek_image', $configuration);
    $elements = $formatter->viewElements($this->items, 'en');

    $this->assertNotEmpty($elements);
    $this->assertEquals('image_formatter', $elements[0]['#theme']);
    $this->assertEquals('medium', $elements[0]['#image_style']);
    $this->assertEquals('pm_12799_15_15886-3he1dvuc8i-whr.jpg', $elements[0]['#item']->entity->getFilename());
    $this->assertEquals('public://keepeek/pm_12799_15_15886-3he1dvuc8i-whr.jpg', $elements[0]['#item']->entity->getFileUri());
  }

  /**
   * Tests that Dynameek image formatter is processed correctly.
   */
  public function testImageDynameek() {
    $configuration = $this->formatterConfiguration;
    $configuration['settings']['image_style'] = 'medium';
    $configuration['settings']['formatter_class'] = KeepeekImageFormatter::class;
    $configuration['settings']['dynameek'] = TRUE;

    /** @var \Drupal\keepeek\Plugin\Field\FieldFormatter\KeepeekImageFormatter $formatter */
    $formatter = $this->container->get('plugin.manager.field.formatter')
      ->createInstance('keepeek_image', $configuration);
    $elements = $formatter->viewElements($this->items, 'en');

    $this->assertNotEmpty($elements);
    $this->assertEquals('image', $elements[0]['#theme']);
    $this->assertEquals('medium', $elements[0]['#style_name']);
    $this->assertEquals('/bXNoxnsb-vX1863wg5UPX6UR8BiylZ5U4dT2Rrraie8/resize?src=kpkp%3A%2F%2Ficoneek%2F12799%2F15%2F15868-w2s35vsrsw.jpg&w=220&h=220&el=false&t=fit', $elements[0]['#uri']);
  }

  /**
   * Tests that responsive image formatter is processed correctly.
   */
  public function testResponsiveImage() {
    $configuration = $this->formatterConfiguration;
    $configuration['settings']['responsive_image_style'] = 'narrow';
    $configuration['settings']['formatter_class'] = KeepeekResponsiveImageFormatter::class;

    /** @var \Drupal\keepeek\Plugin\Field\FieldFormatter\KeepeekResponsiveImageFormatter $formatter */
    $formatter = $this->container->get('plugin.manager.field.formatter')
      ->createInstance('keepeek_responsive_image', $configuration);
    $elements = $formatter->viewElements($this->items, 'en');

    $this->assertNotEmpty($elements);
    $this->assertEquals('responsive_image_formatter', $elements[0]['#theme']);
    $this->assertEquals('narrow', $elements[0]['#responsive_image_style_id']);
    $this->assertEquals('pm_12799_15_15886-3he1dvuc8i-whr.jpg', $elements[0]['#item']->entity->getFilename());
    $this->assertEquals('public://keepeek/pm_12799_15_15886-3he1dvuc8i-whr.jpg', $elements[0]['#item']->entity->getFileUri());
  }

  /**
   * Tests that Dynameek responsive image formatter is processed correctly.
   */
  public function testResponsiveImageDynameek() {
    $configuration = $this->formatterConfiguration;
    $configuration['settings']['responsive_image_style'] = 'narrow';
    $configuration['settings']['formatter_class'] = KeepeekResponsiveImageFormatter::class;
    $configuration['settings']['dynameek'] = TRUE;

    /** @var \Drupal\keepeek\Plugin\Field\FieldFormatter\KeepeekResponsiveImageFormatter $formatter */
    $formatter = $this->container->get('plugin.manager.field.formatter')
      ->createInstance('keepeek_responsive_image', $configuration);
    $elements = $formatter->viewElements($this->items, 'en');

    $this->assertNotEmpty($elements);
    $this->assertEquals('keepeek_responsive_image', $elements[0]['#theme']);
    $this->assertEquals('/vjBArBCab4ee5nXiubNyaxJmZGCG0QjyOnxqU30N72c/resize?src=kpkp%3A%2F%2Ficoneek%2F12799%2F15%2F15868-w2s35vsrsw.jpg&w=325&h=325&el=false&t=fit 1x', $elements[0]['#sources'][0]->toArray()['srcset']);
  }

}
