<?php

namespace Drupal\key_per_language\Plugin\KeyProvider;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\key\Plugin\KeyProviderBase;
use Drupal\key\Plugin\KeyPluginFormInterface;
use Drupal\key\KeyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A key-provider that allows the key to be different per language.
 *
 * @KeyProvider(
 *   id = "key_per_language",
 *   label = @Translation("Key per Language"),
 *   description = @Translation("A key-provider that allows the key to be different per language."),
 *   storage_method = "remote",
 *   key_value = {
 *     "accepted" = FALSE,
 *     "required" = FALSE
 *   }
 * )
 */
class KeyPerLanguageProvider extends KeyProviderBase implements KeyPluginFormInterface {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected LanguageManagerInterface $languageManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->configuration = NestedArray::mergeDeep($this->defaultConfiguration(), $this->configuration);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('language_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'language_type' => LanguageInterface::TYPE_INTERFACE,
      'keys' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {

    $language_types = $this->languageManager->getLanguageTypes();
    $info = $this->languageManager->getDefinedLanguageTypesInfo();

    $types = [];

    foreach ($language_types as $type_key) {
      if (isset($info[$type_key]['name'])) {
        $types[$type_key] = $info[$type_key]['name'];
      }
    }
    $form['language_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Language type'),
      '#description' => $this->t('Select the type of language used to determine the current language of the request.'),
      '#options' => $types,
      '#access' => count($types) > 1,
      '#default_value' => count($types) > 1 ? $this->getConfiguration()['language_type'] : array_key_first($types),
    ];

    $form['keys'] = [
      '#tree' => TRUE,
    ];

    // Get a nice list of the names of all the key providers.
    $key_provider_manager = \Drupal::service('plugin.manager.key.key_provider');
    $providers = $key_provider_manager->getDefinitions();
    // We don't want to allow selecting keys provided by this provider.
    $other_providers = array_diff(array_keys($providers), ['key_per_language']);

    foreach ($this->languageManager->getLanguages() as $langcode => $language) {
      $form['keys'][$langcode] = [
        '#type' => 'key_select',
        '#title' => $this->t('Key for @language', ['@language' => $language->getName()]),
        '#description' => $this->t('Select the key to use for the @language language.', ['@language' => $language->getName()]),
        '#required' => FALSE,
        '#default_value' => $this->getConfiguration()['keys'][$langcode] ?? '',
        '#key_filters' => [
          'provider' => $other_providers,
        ],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {

  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->setConfiguration($form_state->getValues());
  }

  /**
   * {@inheritdoc}
   */
  public function getKeyValue(KeyInterface $key) {
    // Get the current language code.
    $language_code = \Drupal::languageManager()->getCurrentLanguage($this->configuration['language_type'])->getId();

    // Get the key for the current language.
    if (isset($this->configuration['keys'][$language_code])) {
      $key_id = $this->configuration['keys'][$language_code];
      // Pass the key ID to the key repository to load the key.
      $key_entity = \Drupal::service('key.repository')->getKey($key_id);
      if ($key_entity) {
        return $key_entity->getKeyValue();
      }
      else {
        return '';
      }
    }
    else {
      return '';
    }
  }

}
