<?php

namespace Drupal\keycloak\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\keycloak\Service\KeycloakServiceInterface;

/**
 * Provides the Keycloak settings form.
 *
 * @package Drupal\keycloak\Form
 */
class KeycloakSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The OpenID Connect service.
   *
   * @var \Drupal\openid_connect\OpenIDConnect
   */
  protected $openIDConnect;

  /**
   * The OpenID Connect claims service.
   *
   * @var \Drupal\openid_connect\OpenIDConnectClaims
   */
  protected $claims;

  /**
   * The Keycloak service.
   *
   * @var \Drupal\keycloak\Service\KeycloakServiceInterface
   */
  protected KeycloakServiceInterface $keycloak;


  /**
   * The constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *    The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\keycloak\Service\KeycloakServiceInterface $keycloak
   *   The OpenID Connect service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager, EntityTypeManagerInterface $entity_type_manager, KeycloakServiceInterface $keycloak) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->entityTypeManager = $entity_type_manager;
    $this->keycloak = $keycloak;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
      $container->get('keycloak.keycloak'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['keycloak.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'keycloak_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $settings = $this->configFactory()
      ->getEditable('keycloak.settings');

    $keycloak_enabled = $this->keycloak->isEnabled();

    $form['keycloak_sso'] = [
      '#title' => $this->t('Replace Drupal login with Keycloak single sign-on (SSO)'),
      '#type' => 'fieldset',
      '#description' => $this->t("Changes Drupal's authentication back-end to use Keycloak by default. Drupal's user login and registration pages will redirect to Keycloak. Existing users will be able to login using their Drupal credentials at <em>/keycloak/login</em>."),
      '#tree' => TRUE,
      '#disabled' => !$keycloak_enabled
    ];

    $form['keycloak_sso']['enabled'] = [
      '#title' => $this->t('Enabled'),
      '#type' => 'checkbox',
      '#default_value' => ($keycloak_enabled && !empty($settings->get('keycloak_sso')['enabled'])) ? $settings->get('keycloak_sso')['enabled'] : '',
    ];

    if($keycloak_enabled) {
      $keycloak_client_options = array_reduce($this->keycloak->enabledClients(), function($carry, $item) {
          $carry[$item->id()] = $item->label();
          return $carry;
      }, []);

      $form['keycloak_sso']['client'] = [
        '#title' => $this->t('Client'),
        '#type' => 'select',
        '#required' => TRUE,
        '#default_value' => !empty($settings->get('keycloak_sso')['client']) ? $settings->get('keycloak_sso')['client'] : key($keycloak_client_options),
        '#options' => $keycloak_client_options,
        '#description' => $this->t("Select the client to use to override the drupal login path."),
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $this->config('keycloak.settings')
      ->set('keycloak_sso', $form_state->getValue('keycloak_sso'))
      ->save();
  }

}
