<?php

namespace Drupal\keycloak\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\openid_connect\Plugin\OpenIDConnectClientManager;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Keycloak service.
 */
class KeycloakService implements KeycloakServiceInterface {

  /**
   * A list of the enabled keycloak clients
   *
   * @var array
   */
  protected $enabledClients;

  /**
   * The config factory interface.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * A configuration object containing Keycloak client settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Client plugin manager of the OpenID Connect module.
   *
   * @var \Drupal\openid_connect\Plugin\OpenIDConnectClientManager
   */
  protected OpenIDConnectClientManager $oidcClientManager;

  /**
   * A language manager instance.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The current session.
   *
   * @var \Drupal\Core\Session\Session
   */
  protected $session;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;


  /**
   * Default keys to be stored to / retrieved from a Keycloak user session.
   *
   * @var array
   */
  private static $sessionInfoKeys;

  /**
   * Constructor for Drupal\keycloak\Service\KeycloakService.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\openid_connect\Plugin\OpenIDConnectClientManager $oidc_client_manager
   *   Client plugin manager of the OpenID Connect module.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   A language manager instance.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Account proxy for the currently logged-in user.
   * @param \Symfony\Component\HttpFoundation\Session\Session  $session
   *   The current session
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   A logger channel factory instance.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    OpenIDConnectClientManager $oidc_client_manager,
    LanguageManagerInterface $language_manager,
    AccountProxyInterface $current_user,
    Session $session,
    LoggerChannelFactoryInterface $logger,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    $this->config = $config_factory->get('keycloak.settings');
    $this->configFactory = $config_factory;
    $this->oidcClientManager = $oidc_client_manager;
    $this->languageManager = $language_manager;
    $this->currentUser = $current_user;
    $this->session = $session;
    $this->loggerFactory = $logger;
    $this->enabledClients = $entity_type_manager->getStorage('openid_connect_client')->loadByProperties(['plugin' => 'keycloak', 'status' => 1]);
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled() {
    return count($this->enabledClients) > 0;
  }

  /**
   * {@inheritdoc}
   */
  public function enabledClients() {
    return $this->enabledClients;
  }

  /**
   * {@inheritdoc}
   */
  public function getLocale() {
    return $this->config('settings.keycloak_locale_param');
  }

  /**
   * {@inheritdoc}
   */
  public function isKeycloakUser() {
    // Whether the user is not authenticated or the Keycloak client disabled.
    if (!$this->currentUser->isAuthenticated()) {
      return FALSE;
    }

    // If the user was logged in using Keycloak, we will find session
    // information in the current session.
    $tempstore = $this->session->get('keycloak', []);
    return !empty($tempstore[self::OPENID_CONNECT_PLUGIN_ID]);
  }

  /**
   * {@inheritdoc}
   */
  public function getSessionInfoDefaultKeys() {
    if (!isset(self::$sessionInfoKeys)) {
      $default_keys = [
        self::KEYCLOAK_SESSION_ACCESS_TOKEN,
        self::KEYCLOAK_SESSION_REFRESH_TOKEN,
        self::KEYCLOAK_SESSION_ID_TOKEN,
        self::KEYCLOAK_SESSION_CLIENT_ID,
        self::KEYCLOAK_SESSION_SESSION_ID,
        self::OPENID_CONNECT_PLUGIN_ID,
      ];

      self::$sessionInfoKeys = $default_keys;
    }

    return self::$sessionInfoKeys;
  }

  /**
   * {@inheritdoc}
   */
  public function getSessionInfo($keys = NULL) {
    $session_info = [];

    if (!$this->isKeycloakUser()) {
      return $session_info;
    }

    $default_keys = $this->getSessionInfoDefaultKeys();

    $keys = empty($keys) ? $default_keys : array_intersect($default_keys, $keys);
    $tempstore = $this->session->get('keycloak', []);

    foreach ($keys as $key) {
      $session_info[$key] = array_key_exists($key, $tempstore) ? $tempstore[$key] : NULL;
    }

    return $session_info;
  }

  /**
   * {@inheritdoc}
   */
  public function setSessionInfo(array $info) {
    // Whether the user is not authenticated or the Keycloak client disabled.
    if (!$this->currentUser->isAuthenticated() || !$this->isEnabled()) {
      return FALSE;
    }

    $default_keys = $this->getSessionInfoDefaultKeys();
    $old_values = $this->getSessionInfo();
    $new_values = array_merge($old_values, $info);

    $tempstore = $this->session->get('keycloak', []);
    foreach ($default_keys as $key) {
      $tempstore[$key] = $new_values[$key];
    }
    $this->session->set('keycloak', $tempstore);

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function isI18nEnabled() {
    return $this->isEnabled() &&
      $this->languageManager->isMultilingual() &&
      $this->config('settings.keycloak_i18n.enabled');
  }

  /**
   * {@inheritdoc}
   */
  public function getI18nMapping($reverse = FALSE, $include_enabled = TRUE) {
    $mappings = [];

    $languages = $this->languageManager->getLanguages();
    if (empty($languages)) {
      return $mappings;
    }

    $configured = $this->config('settings.keycloak_i18n.mapping');
    // The stored mapping is an unkeyed list of associative arrays
    // with 'langcode' and 'target' as keys. Transform it to an assoc
    // array of 'langcode' => 'target'.
    $kc_mappings = [];
    if (!empty($configured)) {
      foreach ($configured as $mapping) {
        $kc_mappings[$mapping['langcode']] = $mapping['target'];
      }
    }

    // Create the i18n locale mapping information.
    foreach ($languages as $langcode => $language) {
      if (empty($kc_mappings[$langcode]) && !$include_enabled) {
        continue;
      }

      $mapping = [
        'language_id' => $langcode,
        'locale' => !empty($kc_mappings[$langcode]) ? $kc_mappings[$langcode] : $langcode,
        'label' => $language->getName(),
      ];

      $mappings[$reverse ? $mapping['locale'] : $langcode] = $mapping;
    }

    return $mappings;
  }

  /**
   * {@inheritdoc}
   */
  public function isSsoEnabled() {
    return $this->isEnabled() &&
      $this->config->get('keycloak_sso.enabled');
  }

  /**
   * {@inheritdoc}
   */
  public function isKeycloakSignOutEnabled() {
    return $this->config('settings.keycloak_sign_out');
  }

  /**
   * {@inheritdoc}
   */
  public function getKeycloakSignOutEndpoint() {
    return $this->currentClient()->getPlugin()->getEndpoints()['end_session'];
  }

  /**
   * {@inheritdoc}
   */
  public function getKeycloakSignoutResponse(array $session_information) {
    $logout_redirect = Url::fromRoute('<front>', [], ['absolute' => TRUE])->toString();

    if (
      $this->isKeycloakSignOutEnabled() &&
      !empty($session_information[self::KEYCLOAK_SESSION_ID_TOKEN])
    ) {
      // We do an internal redirect here and modify it in
      // our KeycloakRequestSubscriber.
      return new RedirectResponse(Url::fromRoute('keycloak.logout', [], [
        'query' => [
          'id_token_hint' => $session_information[self::KEYCLOAK_SESSION_ID_TOKEN],
          'post_logout_redirect_uri' => $logout_redirect,
        ],
      ])->toString());
    }

    return new RedirectResponse($logout_redirect);
  }

  /**
   * {@inheritdoc}
   */
  public function isCheckSessionEnabled() {
    return $this->isEnabled() &&
      $this->config('settings.check_session.enabled');
  }

  /**
   * {@inheritdoc}
   */
  public function getCheckSessionInterval() {
    return $this->config('settings.check_session.interval');
  }

  /**
   * {@inheritdoc}
   */
  public function getCheckSessionIframeUrl() {
    return $this->currentClient()->getPlugin()->getEndpoints()['session_iframe'];
  }

  /**
   * {@inheritdoc}
   */
  public function getClientInstance() {
    $config = $this->config('settings');

    if (empty($config)) {
      $config = [];
    }

    return $this->oidcClientManager->createInstance(
      'keycloak',
      $config
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getLogger() {
    return $this->loggerFactory->get('openid_connect_keycloak');
  }

  /**
   * {@inheritdoc}
   */
  public function isDebugMode() {
    return $this->config('settings.debug');
  }

  /*
   * Returns the keycloak openid client used for the currently logged in user
   */
  public function currentClient() {
    $tempstore = $this->session->get('keycloak', []);
    if(($plugin_id = array_key_exists(self::OPENID_CONNECT_PLUGIN_ID, $tempstore) ? $tempstore[self::OPENID_CONNECT_PLUGIN_ID] : NULL) && array_key_exists($plugin_id, $this->enabledClients)) {
      return $this->enabledClients[$plugin_id];
    }

    return [];
  }

  /*
   * Returns a configuration item that is dot-targeted. For example settings.keycloak_locale_param
   */
  private function config(string $config) {
    $client = $this->currentClient();
    if(empty($client)) {
      return null;
    }

    $config_levels = explode('.', $config);
    $first_level = array_shift($config_levels);
    $result = $client->get($first_level);

    if (empty($config_levels)) {
      return $result;
    }

    foreach ($config_levels as $level) {
      if (!is_array($result) || !array_key_exists($level, $result)) {
        return null;
      }
      $result = $result[$level];
    }

    return $result;
  }

}
