<?php

namespace Drupal\keycloak_user_provisioning\Form;

use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Url;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Form for displaying upgrade plans and premium features.
 */
class MoKeycloakUpgradePlans extends FormBase {
  use StringTranslationTrait;

  /**
   * The extension list module service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new MoKeycloakUpgradePlans object.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(ModuleExtensionList $module_extension_list, RequestStack $request_stack) {
    $this->moduleExtensionList = $module_extension_list;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('extension.list.module'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId() {
    return 'keycloak_user_provisioning_upgrade_plans_form';
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'keycloak_user_provisioning/keycloak_user_provisioning.admin';

    $base_url = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost();
    $module_path = $base_url . '/' . $this->moduleExtensionList->getPath("keycloak_user_provisioning");
    $module_data = $this->moduleExtensionList->getExtensionInfo('keycloak_user_provisioning');

    $features = [
          [
            'title' => 'Keycloak to Drupal User Sync',
            'description' => 'Set up Drupal to receive user provisioning requests from Keycloak and automatically sync user data from Keycloak to your Drupal site.',
          ],
          [
            'title' => 'Attribute & Role Mapping',
            'description' => 'Map Drupal user attributes and roles with Keycloak attributes and groups to ensure seamless user synchronization and consistent access control between Keycloak and your Drupal site.',
          ],
          [
            'title' => 'Manual/On-Demand Provisioning',
            'description' => 'Manually create, update, or delete users in your Keycloak from Drupal whenever needed with just a click.',
          ],
          [
            'title' => 'Automatic/Event-Based Provisioning',
            'description' => 'Any changes made in Drupal (like adding, editing, or deleting a user) are immediately and automatically applied to Keycloak without manual intervention.',
          ],
          [
            'title' => 'Scheduler-Based Provisioning',
            'description' => 'Keep users synchronized automatically at regular scheduled intervals without any manual work, ensuring consistent data across all systems.',
          ],
    ];

    $related_products = [
          [
            'title' => 'SAML Single Sign-On',
            'description' => 'Engineered for critical infrastructures of Governments, Educational Entities, Healthcare, the Drupal SAML SSO module based on the SAML 2.0 protocol connects to all IDPs prevalent in high security settings like Okta, Azure, miniOrange and even Federal IDPs like Login.gov.',
            'image' => 'saml.webp',
            'link' => 'https://plugins.miniorange.com/drupal-saml-single-sign-on-sso',
          ],
          [
            'title' => 'Drupal OAuth/OIDC Client - SSO',
            'description' => 'The Drupal OAuth/OIDC Client module permits users to perform Single Sign-On (SSO) to your Drupal app via any Identity Providers (IdPs) such as Salesforce, Okta, Azure AD, and more.',
            'image' => 'oauth.webp',
            'link' => 'https://plugins.miniorange.com/drupal-sso-oauth-openid-single-sign-on',
          ],
          [
            'title' => 'REST API Authentication',
            'description' => 'Drupal REST & JSON API Authentication module secures your Drupal site APIs against unauthorized access by enforcing different authentication methods including Basic Authentication, API Key Authentication, JWT Authentication, Third-Party Provider Authentication, etc.',
            'image' => 'REST-API.webp',
            'link' => 'https://plugins.miniorange.com/drupal-rest-api-authentication',
          ],
          [
            'title' => 'Two Factor Authentication',
            'description' => 'Second-Factor Authentication (TFA) adds a second layer of security with an option to configure truly Passwordless Login. You can configure the module to send an OTP to your preferred mode of communication like phone/email, integrate with TOTP Apps like Google Authenticator or configure hardware token 2FA method.',
            'image' => '2FA.webp',
            'link' => 'https://plugins.miniorange.com/drupal-two-factor-authentication-2fa',
          ],

          [
            'title' => 'Session Management',
            'description' => 'User Session Management module helps you to manage the Drupal user session-related operations. It efficiently handles user sessions and provides you with multiple features like terminating any user session from the admin section, auto-logout user on being idle for the configured amount of time, limiting the number of simultaneous sessions per user, IP-based login restrictions, and many more.',
            'image' => 'Session-Management.webp',
            'link' => 'https://plugins.miniorange.com/drupal-session-management',
          ],
          [
            'title' => 'Website Security Pro',
            'description' => 'The Website Security Pro module safeguards your Drupal site with enterprise-grade security. It protects against brute force and DDoS attacks, enforces strong passwords, monitors and blacklists suspicious IPs, and secures login and registration forms. Designed to block hackers and malware, it ensures your site stays secure, stable, and reliable.',
            'image' => 'Web-Security.webp',
            'link' => 'https://plugins.miniorange.com/drupal-web-security-pro',
          ],
    ];

    $form['current_plan_section'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['current-plan-section'],
      ],
    ];
    $form['current_plan_section']['plan_info'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['plan-info']],
    ];

    $form['current_plan_section']['plan_info']['plan_type'] = [
      '#type' => 'markup',
      '#markup' => 'Current Plan: Free Version',
    ];

    $form['current_plan_section']['plan_info']['module_info'] = [
      '#type' => 'markup',
      '#markup' => '<h3>Keycloak User Provisioning</h3>',
      '#attributes' => ['class' => ['module-info']],
    ];

    $form['current_plan_section']['plan_info']['version_info'] = [
      '#type' => 'markup',
      '#markup' => 'Version: ' . $module_data['version'],
    ];

    $form['current_plan_section']['upgrade_button'] = [
      '#type' => 'link',
      '#title' => $this->t('Upgrade Plan'),
      '#url' => Url::fromUri('https://plugins.miniorange.com/drupal-keycloak-user-sync-and-provisioning'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'target' => '_blank',
      ],
    ];

    $form['premium_features_section'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['section-container'],
      ],
    ];

    $form['premium_features_section']['title'] = [
      '#type' => 'markup',
      '#markup' => '<h3>Premium Features</h3>',
      '#prefix' => '<div class="section-title">',
    ];

    $form['premium_features_section']['upgrade-button'] = [
      '#type' => 'link',
      '#title' => $this->t('View All Features'),
      '#url' => Url::fromUri('https://plugins.miniorange.com/drupal-keycloak-user-sync-and-provisioning'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'target' => '_blank',
      ],
      '#suffix' => '</div>',
    ];

    $form['premium_features_section']['features_grid'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['features-grid'],
      ],
    ];

    foreach ($features as $index => $feature) {
      $form['premium_features_section']['features_grid']['feature_' . $index] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['feature-box'],
        ],
      ];

      $form['premium_features_section']['features_grid']['feature_' . $index]['content'] = [
        '#type' => 'markup',
        '#markup' => '<h5>' . $feature['title'] . '</h5>' . $feature['description'],
      ];
    }

    $rows = [
          [
            Markup::create($this->t('<b>1.</b> Click on <a href="https://plugins.miniorange.com/drupal-user-provisioning-and-sync" target="_blank">Upgrade</a> Now button for required licensed plan and you will be redirected to miniOrange login console.</li>')),
            Markup::create($this->t('<b>5.</b> Uninstall and remove the free version of the module from the Drupal module directory')),
          ],
          [
            Markup::create($this->t('<b>2.</b> Enter your username and password with which you have created an account with us. After that you will be redirected to payment page.')),
            Markup::create($this->t('<b>6.</b> Now install the downloaded licensed version of the module.')),
          ],
          [
            Markup::create($this->t('<b>3.</b> Enter your card details and proceed for payment. On successful payment completion, the licensed version module(s) will be available to download.')),
            Markup::create($this->t('<b>7.</b> Clear Drupal Cache from <a href="@base_url" target="_blank">here</a>.', ['@base_url' => $base_url . '/admin/config/development/performance'])),
          ],
          [
            Markup::create($this->t('<b>4.</b> Download the licensed module(s) from Module <a href="https://portal.miniorange.com/downloads" target="_blank">Releases and Downloads</a> section.')),
            Markup::create($this->t('<b>8.</b> After enabling the licensed version of the module, login using the account you have registered with us.')),
          ],
    ];

    $form['markup_how to upgrade'] = [
      '#markup' => '<h3 class="rest-text-center">How to Upgrade to Licensed Version Module</h3>',
    ];

    $form['miniorange_how_to_upgrade_table'] = [
      '#type' => 'table',
      '#responsive' => TRUE,
      '#rows' => $rows,
      '#attributes' => ['class' => ['mo_how_to_upgrade']],
    ];
    $form['related_products_section'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['section-container'],
      ],
    ];

    $form['related_products_section']['title'] = [
      '#type' => 'markup',
      '#markup' => '<h3>Related Products</h3>',
      '#prefix' => '<div class="section-title">',
    ];

    $form['related_products_section']['upgrade-button'] = [
      '#type' => 'link',
      '#title' => $this->t('View All Products'),
      '#url' => Url::fromUri('https://plugins.miniorange.com/drupal'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'target' => '_blank',
      ],
      '#suffix' => '</div>',
    ];

    $form['related_products_section']['features_grid'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['features-grid'],
      ],
    ];

    foreach ($related_products as $index => $product) {
      $form['related_products_section']['features_grid']['feature_' . $index] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['feature-box'],
        ],
      ];

      $form['related_products_section']['features_grid']['feature_' . $index]['title'] = [
        '#type' => 'markup',
        '#markup' => '<h5>' . $product['title'] . '</h5><hr>',
      ];

      $form['related_products_section']['features_grid']['feature_' . $index]['image'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<div><img class="feature-image" src=":module_path/includes/:image" alt=":title" style="width: 400px; height: 400px; object-fit: contain;"></div>', [
          ':module_path' => $module_path,
          ':image' => $product['image'],
          ':title' => $product['title'],
        ]),
      ];

      $form['related_products_section']['features_grid']['feature_' . $index]['content'] = [
        '#type' => 'markup',
        '#markup' => '<span class="product-description">' . $product['description'] . '</span>',
      ];

      $form['related_products_section']['features_grid']['feature_' . $index]['button'] = [
        '#type' => 'link',
        '#title' => $this->t('View Details'),
        '#url' => Url::fromUri($product['link']),
        '#attributes' => [
          'class' => ['button', 'button--primary'],
          'target' => '_blank',
        ],
      ];
    }

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // @todo Implement submitForm() method.
  }

}
