// vite-plugin-sdc-library-generator.ts
import fs from 'fs';
import path from 'path';
import yaml from 'js-yaml';
import fg from 'fast-glob';

/**
 * A plugin to add the libraryOverrides section to component YAML files
 * for SDC components in a Vite build process.
 */
export default function SdcLibraryGenerator(options) {
  const baseDir = options.baseDir ? path.resolve(options.baseDir) : process.cwd();

  return {
    name: 'vite-plugin-sdc-library-generator',
    apply: 'build',

    async buildStart() {
      const entries = await fg('**/*.component.yml', {
        cwd: baseDir,
        absolute: true,
      })
      
      entries.forEach(async (entry) => {
        const folder = path.dirname(entry);
        const folderName = path.basename(folder);

        const cssFileName = `${folderName}.pcss.css`;
        const jsFileName = `${folderName}.es6.js`;
        const cssFilePath = path.join(folder, cssFileName);
        const jsFilePath = path.join(folder, jsFileName);

        let parsed = {};
        const ymlContent = await fs.promises.readFile(entry, 'utf8');

        try {
          parsed = yaml.load(ymlContent, 'utf8') || {};
        } catch (err) {
          this.error(`Failed to parse YAML at ${entry}: ${err}`);
        }

        // Initialize structure if not already present
        parsed.libraryOverrides = parsed.libraryOverrides || {};

        let modified = false;

        if (fs.existsSync(cssFilePath)) {
          if (parsed.libraryOverrides.css === undefined) {
            parsed.libraryOverrides.css = {};
          }
          parsed.libraryOverrides.css.component = {};
          parsed.libraryOverrides.css.component[cssFileName] = {};
          modified = true;
        } else if (parsed.libraryOverrides.css) {
          delete parsed.libraryOverrides.css;
          modified = true;
        }

        if (fs.existsSync(jsFilePath)) {
          if (parsed.libraryOverrides.js === undefined) {
            parsed.libraryOverrides.js = {};
          }
          parsed.libraryOverrides.js[jsFileName] = { attributes: {type: 'module'} };
          modified = true;
        } else if (parsed.libraryOverrides.js) {
          delete parsed.libraryOverrides.js;
          modified = true;
        }

        if (modified) {
          const newYaml = yaml.dump(parsed, { lineWidth: -1 });
          await fs.promises.writeFile(entry, newYaml, 'utf8');
          this.info(`Updated ${folderName}.component.yml with libraryOverrides.`);
        }
      });
    },
  };
}
